import React, { ReactNode, ReactNodeArray } from 'react';

import { useComponents } from 'client/common/hooks';

import cn from 'utils/cn';
import { getDate, humanizeDate } from 'utils/date';
import { reachGoal } from 'utils/metrika';

import { TCustomStyle } from 'utils/findCustomStyle';
import './Media.css';

interface IMediaProps {
    className?: string;
    children: ReactNode;
    slug: string;
    title: string;
    dateString: string;
    isDatesHidden: boolean;
    description: string;
    keywords?: string;
    shareServices: string;
    addonBefore?: ReactNode | ReactNodeArray;
    language?: string;
    shareGoalId?: string;
    imgTitle?: string;
    logo?: string;
    hideShare?: boolean;
    styleVersion?: number;
    customStyles: TCustomStyle[];
}

const cnMedia = cn('media');

function Media({
    slug,
    keywords,
    title,
    dateString,
    description,
    shareServices,
    className,
    addonBefore,
    children,
    language,
    shareGoalId,
    isDatesHidden,
    imgTitle,
    logo,
    hideShare,
    styleVersion,
    customStyles
}: IMediaProps) {
    const { YaShare, SolutionHeader, HtmlContent } = useComponents();

    const date = getDate(dateString) as Date;
    const humanizedDate = humanizeDate(date, 'full', language);

    return (
        <div
            className={cnMedia({}, [className])}
            itemScope
            itemType="http://schema.org/Article"
            >

            <div className={styleVersion === 2 ? cnMedia('block-header') : ''}>
                <div className={cnMedia('block-tags')}>
                    <meta itemProp="identifier" content={slug} />
                    {keywords && <meta itemProp="keywords" content={keywords} />}
                    {addonBefore && (
                    <div className={cnMedia('before')}>{addonBefore}</div>
                    )}

                </div>
                {logo && (<img className={cnMedia('logo')} src={logo} />)}
            </div>

            {
                styleVersion === 2 ?
                    <div className={cnMedia('title-header')}>
                        <SolutionHeader title={title} description={description} imgTitle={imgTitle} customStyles={customStyles} />
                    </div> :
                    <h1
                        className={cnMedia('title')}
                        itemProp="headline"
                        dangerouslySetInnerHTML={{ __html: title }}
                        />
            }

            <div className={cnMedia('line')}>
                {!isDatesHidden && (
                    <div className={cnMedia('date')}>
                        <time dateTime={date.toISOString()} itemProp="dateModified">
                            {humanizedDate}
                        </time>
                    </div>
                )}
                {!hideShare && (<YaShare
                    id={'share-top'}
                    className={cnMedia('share')}
                    popup
                    size="m"
                    services={shareServices}
                    onShare={() => reachGoal(shareGoalId)}
                    />)}
            </div>

            {
                styleVersion === 2 ? '' :
                <HtmlContent
                    className={cnMedia('description')}
                    itemProp="description"
                    content={description}
                    />
            }
            {children}
        </div>
    );
}

export default Media;
