import React, { ChangeEvent, ReactNode, useState } from 'react';
import { useDispatch } from 'react-redux';

import YMap from 'client/common/components/ymap';

import {
    useAgenciesRawFilters,
    useAgencyForm,
    useAgencyOffices,
    useComponents,
    useConfig,
    useLegoComponents
} from 'client/common/hooks';

import { IGeoValue, IOffice, Map } from 'client/common/types';

import {
    addFormOffice,
    changeFormData,
    changeFormOffice,
    removeFormOffice,
    selectOffice
} from 'store/agency/actions';
import { AgencyFormMode } from 'store/agency/types';

import { getFilledEmail } from 'utils/agency';
import cn from 'utils/cn';
import { replaceHtmlEntities } from 'utils/helper';
import { bindKeyset } from 'utils/i18n';
import { reachGoal } from 'utils/metrika';

import './index.css';

export interface IOfficesProps {
    offices: IOffice[];
    site: string;
}

interface ICity extends IGeoValue {
    isCurrent: boolean;
    isMain: boolean;
    officeId: number;
}

const b = cn('offices');
const agencyB = cn('agency');
const i18n = bindKeyset('agency');

const UNMASKED_PHONE_DIGITS_COUNT = 2;

function Offices({ offices, site }: IOfficesProps) {
    const form = useAgencyForm();

    const isFormVisible = form.mode === AgencyFormMode.editing;

    return (
        <div className={b({ mode: form.mode })}>
            {isFormVisible ?
                <OfficesForm /> :
                <Office offices={offices} site={site} />
            }
        </div>
    );
}

interface IOfficeProps {
    offices: IOffice[];
    site: string;
}

function Office({ offices, site }: IOfficeProps) {
    const { Button, Link } = useLegoComponents();
    const { HtmlContent } = useComponents();
    const { htmlMappings } = useConfig();
    const dispatch = useDispatch();

    const officesState = useAgencyOffices();
    const [isPhoneVisible, setIsPhoneVisible] = useState(false);

    function getCities() {
        return offices.reduce((acc: ICity[], office) => {
            const { city } = office;
            const hasOffice = acc.some(item => item.officeId === office.id);

            if (!hasOffice) {
                acc.push({
                    ...city,
                    isCurrent: office.id === officesState.selectedId,
                    isMain: office.isMain,
                    officeId: office.id
                });
            }

            return acc;
        }, []);
    }

    function getMaskedPhone(phone: string) {
        let digitsCount = 0;

        return phone.replace(/\d/g, match => {
            digitsCount += 1;

            return digitsCount > UNMASKED_PHONE_DIGITS_COUNT ? 'X' : match;
        });
    }

    function onShowPhone() {
        reachGoal('agency-click-to-phone');

        setIsPhoneVisible(true);
    }

    function onPhoneClick() {
        reachGoal('agency-click-to-call');
    }

    function onCityClick(officeId: number) {
        return () => {
            setIsPhoneVisible(false);

            dispatch(selectOffice(officeId));
        };
    }

    function onSiteClick() {
        reachGoal('agency-click-to-website');
    }

    function onEmailClick() {
        reachGoal('agency-click-to-email');
    }

    const cities = getCities();
    const isAllCitiesMain = cities.every(city => city.isMain);
    const currentOffice = offices.find(office => office.id === officesState.selectedId);

    if (!currentOffice) {
        return null;
    }

    const {
        phone: rawPhone,
        email,
        address,
        map,
        mapZoom,
        latitude,
        longitude
    } = currentOffice;

    const firstFilledEmail = getFilledEmail(offices);
    const currentEmail = email || firstFilledEmail;

    const phone = replaceHtmlEntities(htmlMappings, rawPhone);
    const coordinates = [latitude, longitude].map(Number) as [number, number];
    const maskedPhone = getMaskedPhone(phone);
    const siteUrl = site.match(/^https?:\/\//) ? site : `https://${site}`;

    return (
        <section className={agencyB('section')}>
            <header className={agencyB('subtitle')}>{i18n('contact-info')}</header>
            <div className={b('office')}>
                <OfficeField type="phone">
                    {isPhoneVisible ?
                        (
                            <MarkedValue>
                                <Link
                                    theme="normal"
                                    href={`tel:${phone}`}
                                    onClick={onPhoneClick}
                                    >
                                    {phone}
                                </Link>
                            </MarkedValue>
                        ) :
                        (
                            <div
                                className={b('masked-phone')}
                                onClick={onShowPhone}
                                >
                                <MarkedValue>{maskedPhone}</MarkedValue>
                                <div className={b('show-phone-button')}>
                                    {i18n('show-phone')}
                                </div>
                            </div>
                        )
                    }
                </OfficeField>

                {currentEmail &&
                    <OfficeField type="email">
                        <MarkedValue>
                            <Link
                                theme="normal"
                                href={`mailto:${currentEmail}`}
                                onClick={onEmailClick}
                                >
                                {currentEmail}
                            </Link>
                        </MarkedValue>
                    </OfficeField>
                }

                <OfficeField type="site">
                    <MarkedValue>
                        <Link
                            theme="normal"
                            href={siteUrl}
                            onClick={onSiteClick}
                            >
                            {site}
                        </Link>
                    </MarkedValue>
                </OfficeField>

                <OfficeField type="cities" i18nKey="city">
                    {cities.map(city => {
                        const { isCurrent, isMain, name, officeId } = city;

                        return (
                            <div
                                key={officeId}
                                className={b('city', {
                                    current: isCurrent,
                                    main: !isAllCitiesMain && isMain
                                })}
                                >
                                <Button
                                    theme="normal"
                                    size="m"
                                    disabled={isCurrent}
                                    onClick={onCityClick(officeId)}
                                    >
                                    {name}
                                </Button>
                            </div>
                        );
                    })}
                </OfficeField>

                <OfficeField type="address">
                    <MarkedValue>
                        <HtmlContent theme="light" content={address} />
                    </MarkedValue>
                </OfficeField>

                <div className={b('map-wrapper')}>
                    <YMap
                        provider={map}
                        zoom={mapZoom}
                        coordinates={coordinates}
                        />
                </div>
            </div>
        </section>
    );
}

function OfficesForm() {
    const form = useAgencyForm();
    const rawFilters = useAgenciesRawFilters();
    const dispatch = useDispatch();

    const { Button, Icon } = useLegoComponents();

    if (!form.data) {
        return null;
    }

    const { offices, site } = form.data;

    function onOfficeAdd() {
        const lastOffice = offices[offices.length - 1] || {};

        const office = {
            email: '',
            phone: '',
            ...lastOffice,
            city: rawFilters.cities[0],
            address: '',
            isMain: false,
            latitude: '',
            longitude: '',
            map: Map.yandex,
            mapZoom: 15
        };

        dispatch(addFormOffice(office));
    }

    return (
        <div className={b('offices-form')}>
            {offices.map((office, index) => {
                function onOfficeRemove() {
                    dispatch(removeFormOffice(index));
                }

                const canRemove = offices.length > 1 && !office.isMain;

                return (
                    // Сделали исключение, так как нет id, чтобы сделать ключом
                    // eslint-disable-next-line react/jsx-key
                    <section className={agencyB('section')}>
                        <header className={agencyB('subtitle')}>
                            {i18n('contact-info')}
                            {canRemove && (
                                <div
                                    className={agencyB('subtitle-control')}
                                    onClick={onOfficeRemove}
                                    >
                                    {i18n('remove')}
                                </div>
                            )}
                        </header>
                        <OfficeForm
                            index={index}
                            office={office}
                            site={site}
                            />
                    </section>
                );
            })}
            <div className={b('add-office-button')}>
                <Button
                    theme="normal"
                    size="m"
                    onClick={onOfficeAdd}
                    >
                    <Icon
                        className={b('add-office-icon')}
                        type="plus"
                        />
                    {i18n('add-office')}
                </Button>
            </div>
        </div>
    );
}

interface IOfficeFormProps {
    index: number;
    office: IOffice;
    site: string;
}

function OfficeForm({ index, office, site }: IOfficeFormProps) {
    const { Textinput, Select } = useLegoComponents();

    const rawFilters = useAgenciesRawFilters();
    const dispatch = useDispatch();

    function onPhoneChange(event: ChangeEvent<HTMLInputElement>) {
        dispatch(changeFormOffice(index, { phone: event.target.value }));
    }

    function onEmailChange(event: ChangeEvent<HTMLInputElement>) {
        dispatch(changeFormOffice(index, { email: event.target.value }));
    }

    function onSiteChange(event: ChangeEvent<HTMLInputElement>) {
        dispatch(changeFormData({ site: event.target.value }));
    }

    function onAddressChange(event: ChangeEvent<HTMLInputElement>) {
        dispatch(changeFormOffice(index, { address: event.target.value }));
    }

    function onCityChange(event: ChangeEvent<HTMLSelectElement>) {
        const geoId = event.target.value;
        const city = rawFilters.cities.find(item => item.geoId === Number(geoId));

        dispatch(changeFormOffice(index, { city }));
    }

    const {
        phone,
        email,
        city,
        address,
        isMain
    } = office;

    const cityOptions = rawFilters.cities.map(item => {
        return {
            content: item.name,
            value: String(item.geoId)
        };
    });

    return (
        <div className={b('office-form')}>
            <OfficeField type="phone">
                <Textinput
                    theme="normal"
                    size="m"
                    value={phone}
                    onChange={onPhoneChange}
                    />
            </OfficeField>

            <OfficeField type="email">
                <Textinput
                    theme="normal"
                    size="m"
                    value={email}
                    onChange={onEmailChange}
                    />
            </OfficeField>

            {index === 0 && (
                <OfficeField type="site">
                    <Textinput
                        theme="normal"
                        size="m"
                        value={site}
                        onChange={onSiteChange}
                        />
                </OfficeField>
            )}

            <OfficeField type="city">
                <Select
                    theme="normal"
                    size="m"
                    disabled={isMain}
                    options={cityOptions}
                    value={String(city.geoId)}
                    onChange={onCityChange}
                    />
            </OfficeField>

            <OfficeField type="address">
                <Textinput
                    theme="normal"
                    size="m"
                    value={address}
                    onChange={onAddressChange}
                    />
            </OfficeField>
        </div>
    );
}

interface IOfficeField {
    type: string;
    i18nKey?: string;
    children: ReactNode;
}

function OfficeField({ type, i18nKey, children }: IOfficeField) {
    return (
        <div className={b('field', { type })}>
            <div className={b('label')}>{i18n(i18nKey || type)}</div>
            <div className={b('value')}>{children}</div>
        </div>
    );
}

interface IMarkedValue {
    children: ReactNode;
}

function MarkedValue({ children }: IMarkedValue) {
    return <div className={b('marked-value')}>{children}</div>;
}

export default Offices;
