import React, { Fragment } from 'react';

import cn from 'utils/cn';

import { useComponents } from 'client/common/hooks';

import { bindKeyset } from 'utils/i18n';

import { IAgencyMetricProductsInfo, IAgencyMetrics, TMetricAgencyKeys } from 'client/common/types';
import { isNumber } from 'lodash';

import { useDispatch } from 'react-redux';
import { showOrHideProductsCard } from 'store/agency/actions';

import './ProductsInfoCard.css';

const i18n = bindKeyset('agency');

const b = cn('products-info-card');

interface IProductsInfoCard{
    agencyName: string;
    metricAgency: IAgencyMetricProductsInfo[];
    metrics: IAgencyMetrics;
    totalMetric: number;
    totalMetricSlug: TMetricAgencyKeys;
}

const ProductsInfoCard = function (props: IProductsInfoCard) {
    const { metricAgency, metrics, agencyName, totalMetric, totalMetricSlug } = props;
    const { HtmlContent } = useComponents();

    const dispatch = useDispatch();

    function setShowOrHideProductsCard(certificate: 'market'| 'direct' | 'metrika' | 'toloka' | 'dialogs', show: boolean) {
        dispatch(showOrHideProductsCard(certificate, show));
    }

    function convertNumberToString(num: number | undefined | boolean, dp: number = 2): string {
        if (!num || !isNumber(num)) {
            return '0.00';
        }

        return num.toFixed(dp);
    }

    function getWidth(currentValue: number, maxValue: number): string {
        const maxWidth = 100;
        const result = (maxWidth / maxValue) * currentValue;

        return `${result}%`;
    }

    return (
        <div className={b()}>
            <button
                className={b('close-button')}
                onClick={() => {
                    setShowOrHideProductsCard('market', false);
                }}
                />
            <div className={b('graphic-view-wrapper')}>
                {
                    metricAgency.map(metric => {
                        const styleAgencyMetric = {
                            width: getWidth(metric.value, metrics[metric.slug].max_value)
                        };
                        const styleMetric = {
                            width: getWidth(metrics[metric.slug].avg_value, metrics[metric.slug].max_value)
                        };

                        if (metric.slug === totalMetricSlug) {
                            return;
                        }

                        return (
                            <Fragment key={`${metric.slug}_${metrics[metric.slug].max_value}`}>
                                <div className={b('metric-name')}>{metrics[metric.slug].title.ru}</div>
                                <div className={b('graphic')}>
                                    <div className={b('graphic-value')}>
                                        <div className={b('graphic-lines')}>
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                        </div>
                                        <div className={b('graphic-view-agency-metric')} style={styleAgencyMetric} />
                                        <span className={metric.value ? b('graphic-value-agency') : b('graphic-value-avg')}>{convertNumberToString(metric.value)}</span>
                                    </div>
                                    <div className={b('graphic-value')}>
                                        <div className={b('graphic-lines')}>
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                            <div className={b('line')} />
                                        </div>
                                        <div className={b('graphic-view-metric')} style={styleMetric} />
                                        <span className={b('graphic-value-avg')}>{convertNumberToString(metrics[metric.slug].avg_value)}</span>
                                    </div>
                                </div>
                            </Fragment>
                        );
                    })
                }
            </div>
            <div className={b('description')}>
                <div className={b('numeric-value')}>
                    <div>
                        <HtmlContent className={b('numeric-label')} content={metrics[totalMetricSlug].title.ru} />
                        <HtmlContent className={b('numeric-media-guru_value')} content={String(totalMetric)} />
                    </div>
                    <div>
                        <HtmlContent className={b('numeric-avg-label')} content={i18n('average-by-market')} />
                        <HtmlContent
                            className={b('numeric-avg_value')}
                            content={convertNumberToString(metrics._total?.active && metrics._total.avg_value)}
                            />
                    </div>
                </div>

                <div className={b('map-history')}>
                    <div className={b('media-guru-color')} />
                    <HtmlContent className={b('media-guru-text')} content={agencyName} />
                    <div className={b('avg-color')} />
                    <HtmlContent className={b('avg-text')} content={i18n('market-average-summary')} />
                </div>
            </div>
        </div>
    );
};

export default ProductsInfoCard;
