import React, { Component, createRef } from 'react';

import { IPhoneData, ISideMenu } from 'client/common/types';
import cn from 'utils/cn';
import debounce from 'utils/debounce';
import i18n from 'utils/i18n';

import RegionSwitcher, { IRegionSwitcherProps } from 'client/common/components/region-switcher';
import { isElementVisible } from 'utils/helper';

import { ComponentRegistryConsumer } from '@bem-react/di';

import MenuItem from './menuItem';

import { ISidebarDispatchProps, ISidebarStateProps } from '.';

import './index.css';

export const b = cn('sidebar');
const TOP_OFFSET = 130;

enum SidebarPosition {
    default = 'default',
    fixedTop = 'fixedTop',
    fixedBottom = 'fixedBottom'
}

function fixedMod(position: SidebarPosition) {
    if (position === SidebarPosition.fixedTop) {
        return { fixed: 'top' };
    }
    if (position === SidebarPosition.fixedBottom) {
        return { fixed: 'bottom' };
    }

    return {};
}

export interface ISidebarOwnProps extends IRegionSwitcherProps {
    phoneData: IPhoneData;
    siteSearchUrl: string;
    menu: ISideMenu;
    withScrollHandling?: boolean;
    isTouch?: boolean;
}

export type ISidebarProps =
    & ISidebarOwnProps
    & ISidebarStateProps
    & ISidebarDispatchProps;

interface ISidebarState {
    viewportHeight: number;
    position: SidebarPosition;
    sidebarListMarginTop: number;
    sidebarListMarginBottom: number;
    hiddenContent: boolean;
    oldMenuIndexChain: number[] | null;
    direction: string;
    sidebarMod: { animated?: 'yes' };
}

class Sidebar extends Component<ISidebarProps, ISidebarState> {
    state = {
        viewportHeight: 0,
        position: SidebarPosition.default,
        sidebarListMarginTop: 0,
        sidebarListMarginBottom: 0,
        hiddenContent: false,
        oldMenuIndexChain: null,
        direction: '',
        sidebarMod: {}
    };

    onResize = debounce(
        () => {
            this.setState({ viewportHeight: window.innerHeight });
        },
        300,
        true
    );
    private readonly listRef = createRef<HTMLDivElement>();
    private readonly listWrapperRef = createRef<HTMLDivElement>();
    private readonly contentRef = createRef<HTMLDivElement>();
    private prevPosition: number = 0;
    private prevTouchPosition: number = 0;

    componentDidMount() {
        if (this.props.withScrollHandling) {
            window.addEventListener('scroll', this.onScroll);
        }

        window.addEventListener('resize', this.onResize);

        if (this.props.isTouch && this.contentRef.current) {
            const content = this.contentRef.current;

            content.addEventListener('touchstart', this.stopScroll);
            content.addEventListener('touchmove', this.stopScroll);
        }
        // eslint-disable-next-line react/no-did-mount-set-state
        this.setState({ viewportHeight: window.innerHeight }, this.updateListWrap);

        this.prevPosition = window.pageYOffset;

        // Не анимируем меню при открытии во время загрузки страницы
        setTimeout(this.setAnimatedMod, 1000);
    }

    componentWillUnmount() {
        if (this.props.withScrollHandling) {
            window.removeEventListener('scroll', this.onScroll);
        }

        window.removeEventListener('resize', this.onResize);

        if (this.props.isTouch && this.contentRef.current) {
            const content = this.contentRef.current;

            content.removeEventListener('touchstart', this.stopScroll);
            content.removeEventListener('touchmove', this.stopScroll);
        }
    }

    setAnimatedMod = () => this.setState({ sidebarMod: { animated: 'yes' } });

    stopScroll = (event: TouchEvent) => {
        const coords = event.touches[0].pageY;

        if (event.type === 'touchstart' || !this.contentRef.current) {
            this.prevTouchPosition = coords;

            return;
        }

        const content = this.contentRef.current;
        const footer = content.querySelector(`.${b('footer')}`);

        const scrollDifference = content.scrollHeight - content.offsetHeight;
        const isScrolledDown = this.prevTouchPosition > coords &&
            content.scrollTop >= scrollDifference;

        const isVisible = isElementVisible(footer!);

        if (isScrolledDown && isVisible) {
            event.preventDefault();
        }

        this.prevTouchPosition = coords;
    }

    onScroll = (event: Event) => {
        event.stopPropagation();

        this.updateListWrap();
        this.prevPosition = window.pageYOffset;
    }

    updateListWrap = () => {
        if (this.isSmallSidebar()) {
            this.updateSmallListWrap();
        } else {
            this.updateBiggerListWrap();
        }
    }

    /**
     * Проверка, что все пункты меню видны на одном экране.
     * @private
     * @returns {Boolean}
     */
    isSmallSidebar = () => {
        if (!this.listWrapperRef.current) {
            return false;
        }

        const { viewportHeight } = this.state;
        const { clientHeight } = this.listWrapperRef.current;
        const paddingTop = this.getListWrapperPadding();

        return clientHeight - paddingTop + TOP_OFFSET <= viewportHeight;
    }

    updateSmallListWrap = () => {
        if (this.state.position !== SidebarPosition.fixedTop) {
            this.setState({ position: SidebarPosition.fixedTop });

            this.setListWrapperPadding(TOP_OFFSET);
        }

        // FIXME: Подумать как сделать лучше (например унести в Redux из шапки)
        const header = document.querySelector<HTMLElement>('.header')!;
        const margin = parseInt(header.style.marginTop || '0', 10);

        this.setListMargin(margin);
    }

    updateBiggerListWrap = () => {
        const direction = this.prevPosition <= window.pageYOffset;

        if (direction) {
            this.handleDownDirection();
        } else {
            this.handleUpDirection();
        }
    }

    setListMargin = (margin: number) => {
        if (!this.isSmallSidebar()) {
            return;
        }

        this.setState({
            sidebarListMarginTop: margin,
            sidebarListMarginBottom: -margin
        });
    }

    handleDownDirection = () => {
        if (this.state.position === SidebarPosition.fixedTop && window.pageYOffset > 0) {
            this.setState({ position: SidebarPosition.default });
            this.setListWrapperPadding(window.pageYOffset + TOP_OFFSET);

            return;
        }

        const listWrapper = this.listWrapperRef.current;

        if (!listWrapper) {
            return;
        }

        const listWrapperHeight = listWrapper.clientHeight;
        const maxHeight = window.pageYOffset + this.state.viewportHeight;

        if (this.state.position !== SidebarPosition.fixedBottom && listWrapperHeight <= maxHeight) {
            this.setState({ position: SidebarPosition.fixedBottom });
            this.setListWrapperPadding(0);
        }
    }

    handleUpDirection = () => {
        const listWrapper = this.listWrapperRef.current;

        if (!listWrapper) {
            return;
        }

        if (this.state.position === SidebarPosition.fixedBottom) {
            const newPadding = (
                window.pageYOffset +
                this.state.viewportHeight -
                listWrapper.clientHeight
            );

            this.setState({ position: SidebarPosition.default });
            this.setListWrapperPadding(newPadding);
        }

        const offset = this.getListWrapperPadding();

        if (this.state.position !== SidebarPosition.fixedTop &&
            offset - TOP_OFFSET > window.pageYOffset
        ) {
            this.setState({ position: SidebarPosition.fixedTop });
            this.setListWrapperPadding(TOP_OFFSET);
        }
    }

    getListWrapperPadding() {
        if (!this.listWrapperRef.current) {
            return 0;
        }

        return parseInt(this.listWrapperRef.current.style.paddingTop || '0', 10);
    }

    /* eslint-disable react/no-did-update-set-state */
    componentDidUpdate(oldProps: ISidebarProps) {
        if (this.props.isOpen && !oldProps.isOpen) {
            this.setState({ hiddenContent: false }, this.updateListWrap);
        }
        const notNull = Boolean(oldProps.menuIndexChain);

        // Нажали бэк у меню
        if (notNull && oldProps.menuIndexChain.length > this.props.menuIndexChain.length) {
            this.setState({ oldMenuIndexChain: oldProps.menuIndexChain, direction: 'left' });
        }

        // Перешли в подраздел без перезагрузки страницы
        if (notNull && oldProps.menuIndexChain.length < this.props.menuIndexChain.length) {
            this.setState({ oldMenuIndexChain: oldProps.menuIndexChain, direction: 'right' });
        }
    }
    /* eslint-enable react/no-did-update-set-state */

    setListWrapperPadding(padding: number) {
        if (!this.listWrapperRef.current) {
            return;
        }

        this.listWrapperRef.current.style.paddingTop = `${padding}px`;
    }

    render() {
        const {
            isOpen,
            menuIndexChain,
            tld,
            regions,
            phoneData,
            siteSearchUrl,
            menu
        } = this.props;
        const {
            position,
            hiddenContent,
            oldMenuIndexChain,
            sidebarListMarginTop,
            sidebarListMarginBottom,
            sidebarMod
        } = this.state;
        const { phone, text } = phoneData;

        const needTempContent = Boolean(oldMenuIndexChain);
        let currentIndexChain = menuIndexChain;

        const needPushNewIndecies = needTempContent && (
            (oldMenuIndexChain || []).length < menuIndexChain.length
        );

        if (needPushNewIndecies) {
            currentIndexChain = oldMenuIndexChain || [];
        }

        const newMod = { active: isOpen ? 'yes' : 'no' };

        return (
            <ComponentRegistryConsumer id={'app'}>
                {({ Search }) => (
                    <div className={b(sidebarMod)}>
                        <div
                            className={b(
                                'content',
                                { hidden: hiddenContent, ...fixedMod(position), ...newMod }
                            )}
                            >
                            <div className={b('list-wrap')} ref={this.listWrapperRef}>
                                <div
                                    className={b('list')}
                                    ref={this.listRef}
                                    style={{
                                        marginTop: sidebarListMarginTop,
                                        marginBottom: sidebarListMarginBottom
                                    }}
                                    >
                                    {this.renderMenu(menu, currentIndexChain)}
                                </div>
                                <div className={b('footer')}>
                                    {siteSearchUrl && (
                                        <div className={b('footer-advertising')}>
                                            <Search siteSearchUrl={siteSearchUrl} type="sidebar" />
                                        </div>
                                    )}
                                    {(phone || text) && (
                                        <div className={b('footer-advertising')}>
                                            {phone && (
                                                <a
                                                    className={b('footer-phone')}
                                                    href={`tel:${phone}`}
                                                    >
                                                    {phone}
                                                </a>
                                            )}
                                            {text && (
                                                <div className={b('phone-description')}>
                                                    {text}
                                                </div>
                                            )}
                                        </div>
                                    )}
                                    <div className={b('footer-location')}>
                                        <RegionSwitcher tld={tld} regions={regions} />
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                )
                }
            </ComponentRegistryConsumer>
        );
    }

    onItemClick = (index: number) => {
        this.props.pushMenuItem(index);
    }

    onBackClick = () => {
        this.props.popMenuItem();
    }

    renderMenu = (menu: ISideMenu, menuIndexChain: number[]) => {
        const curItems = this.getCurrentItems(menu, menuIndexChain);

        const items = curItems.map((item, index) => (
            // eslint-disable-next-line react/no-array-index-key
            <MenuItem key={index} item={item} index={index} onItemClick={this.onItemClick} />
        ));

        if (menuIndexChain.length > 0) {
            const backwardItem = (
                <div className={b('backward')} key="backward" onClick={this.onBackClick}>
                    <div className={b('backward-arrow')} />
                    {i18n({
                        keyset: 'menu',
                        key: 'backward'
                    })}
                </div>
            );

            items.unshift(backwardItem);
        }

        return items;
    }

     getCurrentItems = (menu: ISideMenu, menuIndexChain: number[]) => {
         if (menuIndexChain.length === 0) {
             return menu.items;
         }
         // Показываем итемы предпоследнего активного итема
         let curItems = menu.items;

         for (let i = 0; i < menuIndexChain.length; i += 1) {
             const current = Number(menuIndexChain[i]);
             const currentItem = curItems[current];

             if (!currentItem.items) {
                 return curItems;
             }
             curItems = currentItem.items;
         }

         return curItems;
     }
}

export default Sidebar;
