import React, { Fragment, ReactNode } from 'react';

import { useComponents } from 'client/common/hooks';

import {
    ISolution,
    ISolutionSection,
    TFiltersCategory
} from 'client/common/types';

import cn from 'utils/cn';
import { getDate } from 'utils/date';
import { getTags } from 'utils/solutions';

export interface ISolutionCardProps {
    className?: string;
    solution: ISolution;
    view?: 'default' | 'agency';
    tags?: ReactNode;
    url?: string;
    solutionsUrl?: string;
    mark?: string;
    cardType?: string;
    isDatesHidden?: boolean;
}

export const b = cn('solution-card');

function SolutionCard({
    className = '',
    solution,
    view = 'default',
    tags = null,
    url,
    solutionsUrl = '',
    mark,
    cardType,
    isDatesHidden
}: ISolutionCardProps) {
    const {
        title: solutionTitle,
        theme,
        date: dateString,
        listImage,
        listImageView,
        important,
        cardType: solutionCardType,
        companySize
    } = solution;

    const { Card } = useComponents();

    const date = getDate(dateString);

    return (
        <Card
            classNames={[b(), className]}
            view={view}
            title={solutionTitle}
            url={url || getSolutionUrl(solutionsUrl, solution)}
            date={date}
            mark={mark}
            tags={tags}
            theme={theme}
            companySize={companySize}
            backgroundImage={listImage}
            backgroundImageView={listImageView}
            isImportant={important}
            cardType={cardType || solutionCardType}
            isDatesHidden={isDatesHidden}
            />
    );
}

function getSolutionUrl(solutionsUrl: string, solution: ISolution) {
    const { slug, section } = solution;

    return [
        solutionsUrl,
        section === '_self' ? null : section,
        slug
    ]
        .filter(Boolean)
        .join('/');
}

interface ISolutiоnCardTagsProps {
    solution: ISolution;
    filter: Pick<ISolutionSection, TFiltersCategory>;
    section: string;
    categories: TFiltersCategory[];
    sections: string[];
    solutionsUrl: string;
    onTagClick: (category: TFiltersCategory, tagSlug: string) => void;
}

function SolutionCardTags({
    solution,
    filter,
    section,
    categories,
    sections,
    solutionsUrl,
    onTagClick
}: ISolutiоnCardTagsProps) {
    const { Card } = useComponents();

    const tags = getTags(solution, {
        filter,
        categories,
        addSection: !section
    }).map(({ slug: tagSlug, name: tagName, category }, index) => {
        function handleTagClick() {
            onTagClick(category as TFiltersCategory, tagSlug);
        }

        if (index === 0 && !section && sections.includes(tagSlug)) {
            return (
                <Card.Tag
                    key={tagSlug}
                    name={tagName}
                    url={`${solutionsUrl}/${tagSlug}`}
                    />
            );
        }

        return (
            <Card.Tag
                key={tagSlug}
                name={tagName}
                onClick={handleTagClick}
                />
        );
    });

    return (
        <Fragment>
            {tags}
        </Fragment>
    );
}

SolutionCard.Tags = SolutionCardTags;

export default SolutionCard;
