import url from 'url';

import { IClassNameProps } from '@bem-react/core';
import React, { Component, createRef } from 'react';
import { IYaShareBlock } from 'ya-share';

import cn from 'utils/cn';

import './index.css';

export interface IYaShareProps extends IClassNameProps {
    id: string;
    size: 's' | 'm';
    popup?: boolean;
    lang?: string;
    services?: string;
    blockOrder?: number;
    bare?: boolean;
    onShare?(name: string): void;
}

interface IYaShareState {
    isPopupVisible?: boolean;
}

type TShareMods = Partial<Record<'popup' | 'visible', string>>;

const cnShare = cn('share');

export default class YaShare extends Component<IYaShareProps, IYaShareState> {
    private static constructUrl(urlObject: url.UrlObject, blockOrder?: number) {
        const query = blockOrder ? { block: blockOrder } : {};

        return url.format({ ...urlObject, query });
    }

    state: IYaShareState = {
        isPopupVisible: false
    };

    private yaShareBlock: IYaShareBlock;
    private wrapperRef = createRef<HTMLDivElement>();

    componentDidMount() {
        this.initializeYaShare();

        if (this.props.popup) {
            document.addEventListener('pointerdown', this.handleClickOutside);
        }
    }

    componentDidUpdate(prevProps: IYaShareProps) {
        if (prevProps.id !== this.props.id) {
            this.yaShareBlock.destroy();
            this.initializeYaShare();
        }
    }

    componentWillUnmount() {
        this.yaShareBlock.destroy();

        if (this.props.popup) {
            document.removeEventListener('pointerdown', this.handleClickOutside);
        }
    }

    handleClickOutside = (event: MouseEvent) => {
        event.preventDefault();

        if (!this.wrapperRef!.current!.contains(event.target as Node)) {
            this.setState({ isPopupVisible: false });
        }
    }

    handleClick = () => {
        this.setState({ isPopupVisible: true });
    }

    render() {
        const { id, className, popup } = this.props;

        const { isPopupVisible } = this.state;

        const shareMod: TShareMods = {};

        if (popup) {
            shareMod.popup = 'yes';
        }

        if (isPopupVisible) {
            shareMod.visible = 'yes';
        }

        return (
            <div
                id={id}
                ref={this.wrapperRef}
                className={cnShare(shareMod, [className])}
                onClick={this.handleClick}
                />
        );
    }

    private initializeYaShare() {
        if (!window || !window.Ya || !window.Ya.share2) {
            return;
        }

        const { bare = true, lang, id, size, services, blockOrder } = this.props;
        const { protocol, hostname, pathname } = window.location;

        const shareUrl = YaShare.constructUrl({
            protocol,
            hostname,
            pathname
        }, blockOrder);

        const settings = {
            content: {
                url: shareUrl
            },
            theme: {
                bare,
                size,
                lang,
                services
            },
            hooks: {
                onshare: this.props.onShare
            }
        };

        this.yaShareBlock = window.Ya.share2(id, settings);
    }
}
