import React, { ComponentType, useCallback } from 'react';
import { ArrayParam, useQueryParams } from 'use-query-params';

import { TFiltersCategory } from 'client/common/types';

export function useFilters() {
    const [query, setQuery] = useQueryParams({
        tasks: ArrayParam,
        products: ArrayParam,
        industries: ArrayParam,
        companySize: ArrayParam,
        changes: ArrayParam,
        periods: ArrayParam
    });

    const add = useCallback((category: TFiltersCategory, item: string) => {
        const currentValue = query[category] || [];

        setQuery({
            [category]: currentValue.concat(item)
        });
    }, [query, setQuery]);

    const remove = useCallback((category: TFiltersCategory, item: string) => {
        const currentValue = query[category] || [];
        const newValue = currentValue.filter(element => element !== item);

        setQuery({
            [category]: newValue
        });
    }, [query, setQuery]);

    const removeAll = useCallback(() => {
        setQuery({
            tasks: [],
            products: [],
            industries: [],
            companySize: [],
            changes: [],
            periods: []
        });
    }, [setQuery]);

    return {
        add, remove, removeAll,
        value: query
    };
}

interface IWithFiltersProps {
    filters: ReturnType<typeof withFilters>;
}

export function withFilters<T extends IWithFiltersProps>(Component: ComponentType<T>) {
    return function Wrapped(props: Omit<T, 'filters'>) {
        const filters = useFilters();

        return <Component filters={filters} {...props as T} />;
    };
}
