import React from 'react';

import { IFilter } from 'client/common/components/agencies-filters';
import {
    useAgencies,
    useAgenciesFilters,
    useAgenciesRawFilters,
    useComponents,
    useConfig
} from 'client/common/hooks';
import {
    IAgenciesDefaultFilters,
    IAgenciesFilters,
    IBudgetConfig,
    IEnumValue,
    IGeoValue,
    IServicesEnumValue,
    TRegion
} from 'client/common/types';

import cn from 'utils/cn';
import { bindKeyset } from 'utils/i18n';

import './index.css';

const b = cn('agencies-page');
const i18nAgencies = bindKeyset('agencies');
const i18nCommon = bindKeyset('common');

export interface IAgenciesPageProps {
    budgetConfig: IBudgetConfig;
    defaultFilters: IAgenciesDefaultFilters;
    description: string;
    filtersTemplate: string;
    filterUrl: string;
    regions: TRegion[];
    title: string;
    tld: string;
}

function countryToFilterOption(val: IGeoValue) {
    return {
        id: String(val.geoId),
        name: val.name,
        preposition: val.preposition,
        namePrepositional: val.namePrepositional,
        relatedId: null
    };
}

function cityToFilterOption(val: IGeoValue) {
    const { country } = val;

    return {
        id: String(val.geoId),
        name: val.name,
        preposition: val.preposition,
        namePrepositional: val.namePrepositional,
        relatedId: country ? String(country.geoId) : null
    };
}

function enumToFilterOption(val: IEnumValue) {
    return {
        id: val.code,
        name: val.name,
        preposition: val.preposition,
        namePrepositional: val.namePrepositional,
        relatedId: null
    };
}

function servicesToFilterOption(val: IServicesEnumValue) {
    return {
        id: val.code,
        name: val.title,
        certificates: val.certificates,
        preposition: val.preposition,
        namePrepositional: val.namePrepositional,
        relatedId: null
    };
}

function buildFilters(rawFilters: IAgenciesFilters, defaultFilters: IAgenciesDefaultFilters, checkedCertificate: string[]): IFilter[] {
    const filters: IFilter[] = [];

    if (rawFilters.countries.length > 1) {
        filters.push({
            id: 'country',
            options: rawFilters.countries.map(countryToFilterOption),
            allowMultiple: false,
            relatedFilter: {
                filterId: 'city',
                value: defaultFilters.city
            }
        });
    }

    if (rawFilters.cities.length > 0) {
        const cityOptions = rawFilters.cities.map(cityToFilterOption);

        cityOptions.push({
            id: 'all',
            name: i18nAgencies('filter-all-cities-name'),
            namePrepositional: i18nAgencies('filter-all-cities-name-prepositional'),
            preposition: i18nAgencies('filter-all-cities-preposition'),
            relatedId: 'all'
        });

        filters.push({
            id: 'city',
            options: cityOptions
        });
    }

    if (rawFilters.certificates.length > 0) {
        filters.push({
            id: 'certificate',
            options: rawFilters.certificates.map(enumToFilterOption)
        });
    }

    if (rawFilters.services.length > 0) {
        filters.push({
            id: 'service',
            options: rawFilters.services.map(servicesToFilterOption).filter(item => {
                let result = false;

                for (let i = 0; i < checkedCertificate.length; i++) {
                    result = item.certificates ? item.certificates.includes(checkedCertificate[i]) : true;
                    if (!result) {
                        return result;
                    }
                }

                return result;
            })
        });
    }

    return filters;
}

export default function AgenciesPage(props: IAgenciesPageProps) {
    const {
        description,
        filterUrl,
        filtersTemplate,
        defaultFilters,
        budgetConfig,
        tld,
        regions
    } = props;

    const { country: checkedCountry, certificate: checkedCertificate } = useAgenciesFilters();
    const rawFilters = useAgenciesRawFilters();
    const { agenciesConfig: { groupsByCountryGeoId } } = useConfig();
    const { lists, isError } = useAgencies();

    const {
        AgenciesFilters,
        AgenciesList,
        MainLayout,
        AgenciesTitle
    } = useComponents();

    const filters = buildFilters(rawFilters, defaultFilters, checkedCertificate);

    if (checkedCountry) {
        const cityFilter = filters.find(filter => filter.id === 'city')!;

        cityFilter.options = cityFilter.options.filter(option => {
            return option.relatedId === 'all' ||
                   option.relatedId === checkedCountry;
        });
    }

    const otherGroup = groupsByCountryGeoId[checkedCountry!] || groupsByCountryGeoId.default;

    const isMainListEnded = !lists.main.hasMore;
    const isOtherListFullness = lists[otherGroup].agencies.length !== 0;

    const isOtherAgenciesVisible = isMainListEnded && isOtherListFullness;
    const resetTextKey = isOtherAgenciesVisible ? `only-${otherGroup}-found` : 'not-found';

    return (
        <React.Fragment>
            <AgenciesTitle filtersTemplate={filtersTemplate} />
            <MainLayout>
                <AgenciesFilters
                    className={b('filters')}
                    tld={tld}
                    regions={regions}
                    filtersTemplate={filtersTemplate}
                    description={description}
                    filters={filters}
                    budgetConfig={budgetConfig}
                    />
                <MainLayout.Columns>
                    <MainLayout.Content>
                        <AgenciesList
                            visible
                            defaultFilters={defaultFilters}
                            filterUrl={filterUrl}
                            group="main"
                            resetTextKey={resetTextKey}
                            />
                        <div className={b('other-regions', { visible: isOtherAgenciesVisible })}>
                            <div className={b('other-title')}>{i18nAgencies(`other-${otherGroup}`)}</div>
                            <AgenciesList
                                visible={isOtherAgenciesVisible}
                                defaultFilters={defaultFilters}
                                filterUrl={filterUrl}
                                group={otherGroup}
                              //  agenciesInfo={agenciesInfo}
                                />
                        </div>
                        {isError && (
                            <div className={b('loading-error')}>
                                {i18nCommon('loading-error')}
                            </div>
                        )}
                    </MainLayout.Content>
                </MainLayout.Columns>
            </MainLayout>
        </React.Fragment>
    );
}
