import React, {
    Fragment,
    useEffect,
    useReducer
} from 'react';
import Helmet from 'react-helmet';
import { useDispatch } from 'react-redux';
import { useParams } from 'react-router-dom';

import {
    useComponents,
    useConfig,
    useLegoComponents
} from 'client/common/hooks';

import { reachTime } from 'utils/metrika';
import * as scroll from 'utils/scroll';

import { IOffice } from 'client/common/types';
import { selectOffice } from 'store/agency/actions';

import { getInitialState, reducer } from './reducer';
import { ActionType, IAgencyRouteParams, IFetchAgencyOpts, IProps } from './types';

function fetchAgency({ slug, secretKey, agencyUrl }: IFetchAgencyOpts) {
    return fetch(agencyUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': secretKey
        },
        body: JSON.stringify({ slug })
    })
        .then(response => response.json());
}

function AgencyPage(props: IProps) {
    const {
        Agency,
        MainLayout
    } = useComponents();
    const { Spin } = useLegoComponents();

    const { secretKey } = useConfig();
    const { slug } = useParams<IAgencyRouteParams>();

    const [state, dispatch] = useReducer(reducer, getInitialState(props));
    const { isLoading, agency, agenciesPageUrl } = state;
    const { agencyUrl, metrics } = props;
    const seo = agency && agency.seo;

    const reduxDispatch = useDispatch();

    useEffect(() => {
        scroll.to({ top: 0, left: 0 });

        reachTime('vremya');
    }, []);

    useEffect(() => {
        if (agency || isLoading) {
            return;
        }

        dispatch({ type: ActionType.StartLoading });

        fetchAgency({ slug, secretKey, agencyUrl })
            .then(response => {
                dispatch({
                    type: ActionType.FinishLoading,
                    payload: { agency: response.agency }
                });

                const { offices = [] } = response.agency;
                const initialOffice = offices.find((office: IOffice) => office.isMain) || offices[0];

                if (initialOffice) {
                    reduxDispatch(selectOffice(initialOffice.id));
                }
            });
    }, [agency, props, secretKey, slug, isLoading, agencyUrl, reduxDispatch]);

    return (
        <Fragment>
            {seo && (
                <Helmet
                    title={seo.pageTitle}
                    meta={[{ name: 'description', content: seo.description }]}
                    />
            )}
            <MainLayout>
                <MainLayout.Columns transparent>
                    <MainLayout.Content>
                        {isLoading && (
                            <div style={{ height: '400px', margin: '0 auto' }}>
                                <Spin
                                    progress
                                    position="center"
                                    view="default"
                                    size="l"
                                    />
                            </div>
                        )}
                        {!isLoading && agency && (
                            <Agency
                                metrics={metrics}
                                agency={agency}
                                agencyUrl={agencyUrl}
                                agenciesPageUrl={agenciesPageUrl}
                                />
                        )}
                    </MainLayout.Content>
                </MainLayout.Columns>
            </MainLayout>
        </Fragment>
    );
}

export default AgencyPage;
