import React from 'react';

import { useComponents } from 'client/common/hooks';
import {
    IOpenGraph,
    ISolution,
    ISolutionSection,
    TFiltersCategory,
    TLevel
} from 'client/common/types';

import cn from 'utils/cn';
import { reachGoal } from 'utils/metrika';
import { getTags } from 'utils/solutions';

import './index.css';

export interface ISolutionPageProps {
    solution: ISolution;
    solutionsUrl: string;
    filter: Pick<ISolutionSection, TFiltersCategory>;
    keywords?: string;
    levels?: TLevel[];
    shareServices: string;
    language?: string;
    isDatesHidden: boolean;
    imgTitle?: string;
    logo?: string;
    styleVersion?: number;
    hideShare?: boolean;
}

const CATEGORIES: TFiltersCategory[] = [
    'tasks',
    'industries',
    'products',
    'companySize',
    'changes',
    'periods'
];

export const b = cn('solution-info');

function SolutionPage({
    keywords,
    solution,
    solutionsUrl,
    filter,
    levels,
    shareServices,
    language,
    isDatesHidden
}: ISolutionPageProps) {
    const { MainLayout, Level, YaShare, Media } = useComponents();

    const {
        slug,
        title,
        date,
        description,
        shareText,
        topShareGoalId,
        bottomShareGoalId,
        imgTitle,
        logo,
        styleVersion,
        hideShare,
        customStyles
    } = solution;

    const tags = getTags(solution, {
        filter,
        categories: CATEGORIES
    })
        .map(({ slug: tagSlug, name, category }) => (
            <a
                key={tagSlug}
                className={b('tag')}
                href={`${solutionsUrl}?${category}=${tagSlug}`}
                >
                {name}
            </a>
        ));

    interface ITLevelProps {
        levels: TLevel[];
        className?: string;
    }

    interface ITLevelPresetProps {
        level: TLevel;
        className?: string;
        index: number;
    }

    function Levels(props: ITLevelProps) {
        let compactOrder: 'odd' | 'even' | undefined;

        function LevelPreset({ index, className, level }: ITLevelPresetProps) {
            let tagName;

            if (level.type === 'results' && level.tag) {
                const { tag: rawTag } = level;
                const [category, tagSlug] = rawTag.split(':');

                const foundedFilter = filter[category as TFiltersCategory]
                    .find(el => el && el.slug === tagSlug);

                tagName = (foundedFilter && foundedFilter.name) || '';
            }

            const { og, compact } = level;

            compactOrder = (compact || undefined) &&
                (compactOrder === 'odd' ? 'even' : 'odd');

            return (
                // eslint-disable-next-line react/no-array-index-key
                <React.Fragment key={`${level.order}-${index}`}>
                    <Level
                        className={`${b('level')} ${className}`}
                        {...level}
                        tag={tagName}
                        compactOrder={compactOrder}
                        />

                    {og && Object
                        .keys(og)
                        .map((key: keyof IOpenGraph) => og[key])
                        .some(Boolean) && (
                            <YaShare
                                id={`block-${level.order}`}
                                blockOrder={level.order}
                                className={b('share')}
                                popup
                                size="m"
                                services={shareServices}
                                onShare={() => reachGoal(level.shareGoalId)}
                                />
                    )}
                </React.Fragment>
            );
        }

        const groups: TLevel[][] = [];
        const readyGroups = props.levels.reduce(
            (processedGroups, currentValue) => {
                const currentGroup = processedGroups.length ? processedGroups[processedGroups.length - 1] : undefined;
                const previousValue = currentGroup ? currentGroup[currentGroup.length - 1] : undefined;

                if (currentGroup && currentValue.compact && previousValue && previousValue.compact) {
                    currentGroup.push(currentValue);
                } else {
                    processedGroups.push([currentValue]);
                }

                return processedGroups;
            }
            , groups);

        return (
            <div className={`${b('levels')} wraper-${props.className}`} itemProp="articleBody">
                {
                    readyGroups.map((groupLevels, index) => {
                        return (
                            groupLevels.length === 1 ?
                                // eslint-disable-next-line react/no-array-index-key
                                <LevelPreset level={groupLevels[0]} index={index} className={props.className} key={index} /> :
                                <div className={b('compact-group')}>
                                    {
                                        groupLevels.map(level => {
                                            return (
                                                // eslint-disable-next-line react/no-array-index-key
                                                <LevelPreset level={level} index={index} className={props.className} key={index} />
                                            );
                                        })
                                    }
                                </div>
                        );
                    })
                }
            </div>
        );
    }

    interface IShareButtonProps {
        shareText: string;
    }

    function ShareButton(props: IShareButtonProps) {
        return (
            <div className={b('share-bottom')}>
                {props.shareText && (
                    <div className={b('share-text')}>
                        {props.shareText}
                    </div>
                )}
                <YaShare
                    id={'share-bottom'}
                    className={b('share')}
                    size="m"
                    services={shareServices}
                    onShare={() => reachGoal(bottomShareGoalId)}
                    />
            </div>);
    }

    return (
        <MainLayout>
            <MainLayout.Columns>
                <MainLayout.Content>
                    <Media
                        language={language}
                        className={b()}
                        slug={slug}
                        keywords={keywords}
                        title={title}
                        dateString={date}
                        isDatesHidden={isDatesHidden}
                        description={description}
                        shareServices={shareServices}
                        addonBefore={<div className={b('tags')}>{tags}</div>}
                        shareGoalId={topShareGoalId}
                        imgTitle={imgTitle}
                        logo={logo}
                        styleVersion={styleVersion}
                        hideShare={hideShare}
                        customStyles={customStyles}
                        >
                        {levels && (<Levels className={styleVersion === 2 ? b('level-for-mediaV2') : ''} levels={levels} />)}
                        {!hideShare && (<ShareButton shareText={shareText} />)}
                    </Media>
                </MainLayout.Content>
            </MainLayout.Columns>
        </MainLayout>
    );
}

export default SolutionPage;
