import React, {
    createRef,
    useCallback,
    useEffect,
    useReducer,
    useRef
} from 'react';
import cn from 'utils/cn';
import * as scroll from 'utils/scroll';

import SolutionCard from 'client/common/components/solution-card';

import {
    useComponents,
    useFilters,
    useLegoComponents
} from 'client/common/hooks';
import {
    ISolution,
    ISolutionSection,
    TFiltersByCategory,
    TFiltersCategory
} from 'client/common/types';
import i18n from 'utils/i18n';

import { ActionTypes } from './actions';
import { reducer } from './reducer';

import './index.css';

export const b = cn('solutions-list');

const SECTIONS = ['stories', 'cases', 'analytics', 'practicums'];
const CATEGORIES: TFiltersCategory[] = ['industries'];
const CHUNK_LENGTH = 8;

export interface ISolutionsPageProps {
    title: string;
    text: string;
    solutionsList: ISolution[];
    hasMore: TFiltersByCategory;
    filterUrl: string;
    solutionsUrl: string;
    filter: Pick<ISolutionSection, TFiltersCategory>;
    section: string;
    language?: string;
    isDatesHidden: boolean;
}

export interface ISolutionsPageState {
    solutionsList: ISolution[];
    solutionsCount: number;
    more: boolean;
    hasMore: TFiltersByCategory;
    loading: boolean;
    loadingError: boolean;
}

function SolutionsPage({
    title,
    text,
    solutionsList: initialList,
    hasMore: initialHasMore,
    filterUrl,
    solutionsUrl,
    filter,
    section,
    isDatesHidden
}: ISolutionsPageProps) {
    const [state, dispatch] = useReducer(
        reducer,
        {
            solutionsList: initialList,
            solutionsCount: initialList.length,
            hasMore: initialHasMore,
            more: true,
            loading: false,
            loadingError: false
        });

    const itemsRef = createRef<HTMLDivElement>();

    const filters = useFilters();
    const prevFiltersValueRef = useRef(filters.value);

    const { Level } = useComponents();
    const { Spin } = useLegoComponents();

    const { solutionsList, solutionsCount, more, hasMore, loading, loadingError } = state;

    const loadSolutionsChunk = useCallback(() => {
        fetch(filterUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                startIdx: solutionsCount,
                filter: Object.keys(filters.value)
                    .map((category: TFiltersCategory) => ({
                        by: category,
                        value: filters.value[category]
                    }))
                    .filter(item => item.value),
                section
            })
        })
            .then(res => res.json())
            .then(data => {
                if (!data || !Array.isArray(data.list)) {
                    return dispatch({ type: ActionTypes.SET_ERROR });
                }

                dispatch({
                    type: ActionTypes.PUSH_LIST,
                    payload: {
                        list: data.list,
                        more: data.more,
                        hasMore: data.hasMore
                    }
                });
            })
            .catch(() => dispatch({ type: ActionTypes.SET_ERROR }));
    }, [filterUrl, filters.value, section, solutionsCount]);

    useEffect(() => {
        function onScroll() {
            const itemsTopOffset = itemsRef.current!.getBoundingClientRect().top + window.pageYOffset;
            const itemsHeight = itemsRef.current!.clientHeight;

            const windowOffset = window.pageYOffset + window.innerHeight;
            const contentOffset = itemsTopOffset + itemsHeight;

            const areManySolutions = solutionsCount >= CHUNK_LENGTH;

            if (!loading && more && areManySolutions && windowOffset > contentOffset) {
                dispatch({ type: ActionTypes.START_LOADING });
            }
        }

        window.addEventListener('scroll', onScroll);

        return () => window.removeEventListener('scroll', onScroll);
    }, [itemsRef, loadSolutionsChunk, loading, more, solutionsCount]);

    useEffect(() => {
        if (loading) {
            loadSolutionsChunk();
        }
    }, [loading, loadSolutionsChunk]);

    useEffect(() => {
        if (filters.value !== prevFiltersValueRef.current) {
            prevFiltersValueRef.current = filters.value;

            dispatch({ type: ActionTypes.RESET_LIST });
        }
    }, [filters.value, loadSolutionsChunk]);

    const addHandler = (category: TFiltersCategory, item: string) => {
        const categoryFilters = filters.value[category] || [];

        if (!categoryFilters.includes(item)) {
            scroll.to(
                { top: 0, left: 0, behavior: 'smooth' },
                () => filters.add(category, item)
            );
        }
    };

    return (
        <div className={b()}>
            <div className={b('filters')}>
                <Level
                    type="filters"
                    groups={filter}
                    hasMore={hasMore}
                    order={['tasks', 'industries', 'products', 'companySize', 'changes', 'periods']}
                    title={title}
                    text={text}
                    />
            </div>
            <div ref={itemsRef} className={b('items')}>
                {solutionsList && solutionsList.map(solution => {
                    return (
                        <SolutionCard
                            className={b('card')}
                            key={solution.slug}
                            solution={solution}
                            solutionsUrl={solutionsUrl}
                            isDatesHidden={isDatesHidden}
                            tags={(
                                <SolutionCard.Tags
                                    solution={solution}
                                    filter={filter}
                                    section={section}
                                    categories={CATEGORIES}
                                    sections={SECTIONS}
                                    solutionsUrl={solutionsUrl}
                                    onTagClick={addHandler}
                                    />
                            )}
                            />
                    );
                })}
            </div>
            <div className={b('spin')}>
                {loadingError ? (
                    <div className={b('loading-error')}>
                        {i18n({ keyset: 'common', key: 'loading-error' })}
                    </div>
                ) : (
                    <Spin
                        progress={loading}
                        view="default"
                        size="m"
                        position="center"
                        />
                )}
            </div>
        </div>
    );
}

export default SolutionsPage;
