import { ComponentRegistryConsumer } from '@bem-react/di';
import React, { Component, RefObject } from 'react';

import Logo from 'client/common/components/logo';
import cn, { mixHoverOpacity } from 'utils/cn';

import { ConfigContext } from 'client/common/contexts';

import { getHeaderMenu, IHeaderMenuItem } from 'utils/mainMenu';

import 'client/common/components/header';
import './index.css';

interface IHeaderState {
    opened: boolean;
}

const cnHeader = cn('header');

export class Header extends Component<{ headerRef: RefObject<HTMLElement> }, IHeaderState> {
    static contextType = ConfigContext;

    context!: React.ContextType<typeof ConfigContext>;
    state = { opened: false };

    componentDidMount() {
        window.addEventListener('scroll', this.closeMenu);
        window.document.addEventListener('keyup', this.onKeyUp);
    }

    componentWillUnmount() {
        window.removeEventListener('scroll', this.closeMenu);
        window.removeEventListener('keyup', this.onKeyUp);
    }

    render() {
        const { headerRef } = this.props;
        const { mordaHost, language, tld, homeUrl } = this.context;
        const { opened } = this.state;

        const levels = getHeaderMenu(tld, homeUrl);

        return (
            <ComponentRegistryConsumer id={'lego'}>
                {({ User }) => (
                    <header ref={headerRef} className={cnHeader({ opened: opened ? 'yes' : 'no' })}>
                        <div className={cnHeader('line-wrap')}>
                            <div className={cnHeader('current-line')}>
                                <div className={cnHeader('logo')}>
                                    <Logo url={mordaHost} language={language} />
                                </div>
                                <div className={cnHeader('menu')}>
                                    { this.renderLevels(levels) }
                                </div>

                                <div className={cnHeader('right-column')}>
                                    <User />
                                    {
                                        levels.length > 1 && (<div
                                            className={mixHoverOpacity(cnHeader('menu-button'))}
                                            onClick={this.onMenuButtonClick}
                                            />
                                        )
                                    }
                                </div>
                                <div className={cnHeader('paranja')} onClick={this.closeMenu} />
                            </div>
                        </div>
                    </header>
                )}
            </ComponentRegistryConsumer>
        );
    }

    renderLevels = (levels: IHeaderMenuItem[][]) => {
        return levels.map((services, idx: number) => {
            let isCurrentLevel = false;

            const links = services.map(service => {
                if (service.current) {
                    isCurrentLevel = true;
                }

                const { current, url, name } = service;

                return this.renderServiceLink(current, url, name);

            });

            const position = idx.toString();

            const mods = {
                pos: position,
                current: isCurrentLevel
            };

            return (
                <div key={position} className={cnHeader('menu-level', mods)}>
                    {links}
                </div>
            );
        });
    }

    renderServiceLink = (current: boolean, url: string = '', name: string) => {
        const mod = current ? { current: true } : {};
        const className = mixHoverOpacity(cnHeader('link', mod));

        if (url) {
            return <a key={name} className={className} href={url}>{name}</a>;
        }

        return <span key={name} className={className}>{name}</span>;

    }

    onMenuButtonClick = () => {
        this.setState(({ opened }) => ({ opened: !opened }));
    }

    closeMenu = () => {
        this.setState({ opened: false });
    }

    onKeyUp = (event: KeyboardEvent) => {
        if (event.key === 'Escape') {
            this.closeMenu();
        }
    }

}

export default Header;
