import { withBemMod } from '@bem-react/core';
import React, {
    ChangeEventHandler,
    Component,
    createRef,
    FormEvent,
    KeyboardEventHandler
} from 'react';

import { b, ISearchProps } from 'client/common/components/search';
import i18n from 'utils/i18n';

import './search_type_sidebar.css';

export interface ISearchTypeSidebarProps {
    type?: 'sidebar';
}

interface ISearchTypeSidebarState {
    text: string;
    hidden: boolean;
}

export const withSearchTypeSidebar = withBemMod<ISearchTypeSidebarProps, ISearchProps>(
    b(),
    { type: 'sidebar' },
    BaseSearch => {
        return class extends Component<ISearchProps, ISearchTypeSidebarState> {
            state = {
                text: '',
                hidden: false
            };

            inputRef = createRef<HTMLInputElement>();

            onFocus = () => this.hideIcon();

            onBlur = () => this.showIcon();

            hideIcon = () => this.setState({ hidden: true });

            showIcon = () => this.setState({ hidden: false });

            blurInput = () => this.inputRef.current!.blur();

            changeLocation = () => {
                const { text } = this.state;

                const encodedValue = encodeURIComponent(text);

                window.location.assign(`${this.props.siteSearchUrl}&text=${encodedValue}`);
            }

            onSubmit = (event: FormEvent) => {
                event.preventDefault();

                const { text } = this.state;

                if (text) {
                    this.changeLocation();
                }
            }

            onInputChange: ChangeEventHandler<HTMLInputElement> = event => {
                const input = event.target;
                const text = input.value;

                this.setState({ text });
            }

            onKeyDown: KeyboardEventHandler = event => {
                if (event.key === 'Escape') {
                    this.blurInput();
                }
            }

            render() {
                const { text, hidden } = this.state;
                const { className } = this.props;

                const placeholder = i18n({
                    keyset: 'search',
                    key: 'search'
                });

                return (
                    <BaseSearch
                        className={className}
                        value={text}
                        placeholder={placeholder}
                        controlRef={this.inputRef}
                        onKeyDown={this.onKeyDown}
                        onChange={this.onInputChange}
                        onFocus={this.onFocus}
                        onBlur={this.onBlur}
                        onSubmit={this.onSubmit}
                        addonInside={(
                            <div className={b('icon', { hidden })} />
                        )}
                        />
                );
            }
        };
    });
