import 'client/common/polyfill';

import React, { useState } from 'react';
import Helmet from 'react-helmet';

import { cnTheme } from '@yandex-lego/components/Theme';
import { theme } from '@yandex-lego/components/Theme/presets/default';

import cn from 'utils/cn';

import { IStatvalueAppData } from 'client/common/types';

import Meta from 'client/common/components/meta';
import { YMInitializer } from 'client/common/components/yandex-metrika';
import { CommonDataContextProvider, ConfigContextProvider } from 'client/common/contexts';

import { AccessRequest } from 'client/platforms/statvalue/components/access-request';
import { Footer } from 'client/platforms/statvalue/components/footer';
import { Header } from 'client/platforms/statvalue/components/header';
import { Segments } from 'client/platforms/statvalue/components/segments';
import { Survey } from 'client/platforms/statvalue/components/survey';
import { TextBlock } from 'client/platforms/statvalue/components/text-block';
import { Title } from 'client/platforms/statvalue/components/title';
import { ToDirect } from 'client/platforms/statvalue/components/to-direct';

import 'client/common/styles/fonts.css';

import './index.css';

const b = cn('app');

interface IStatvalueAppProps {
    data: IStatvalueAppData;
}

export function App({ data }: IStatvalueAppProps) {
    const {
        calculator,
        commonData,
        domains,
        faviconUrl,
        og,
        popup,
        published,
        seo,
        shareScript,
        survey,
        texts,
        tld
    } = data;

    const [calculated, setCalculated] = useState(false);

    function handleCalculate() {
        setCalculated(true);
    }

    return (
        <ConfigContextProvider data={data}>
            <CommonDataContextProvider data={data}>
                <Meta uatraits={{ isMobile: false }} seo={seo} og={og} />
                <Helmet>
                    <html lang={commonData.language} />
                    <link rel="shortcut icon" href={faviconUrl} />
                    <link
                        rel="stylesheet"
                        href="https://yastatic.net/s3/vda/i/simplebar/5.0.3/simplebar.min.css"
                        />
                </Helmet>
                <Header
                    buttonText={texts.header.buttonText}
                    domains={domains}
                    linkText={texts.header.linkText}
                    published={published}
                    textBlocks={texts.textBlocks}
                    tld={tld}
                    />
                <div className={b({}, [cnTheme(theme)])}>
                    <Title
                        className={b('title')}
                        title={texts.title.title}
                        subtitle={texts.title.subtitle}
                        />
                    <Segments
                        className={b('segments')}
                        calculator={calculator}
                        onCalculate={handleCalculate}
                        />
                    {texts.textBlocks.map(textBlock => (
                        <TextBlock
                            key={textBlock.title}
                            className={b('text-block')}
                            id={textBlock.anchor}
                            text={textBlock.text}
                            title={textBlock.title}
                            />
                    ))}
                    {published ? (
                        <ToDirect
                            className={b('to-direct')}
                            buttonText={texts.toDirect.buttonText}
                            link={texts.toDirect.link}
                            title={texts.toDirect.title}
                            />
                    ) : (
                        <AccessRequest
                            className={b('access-request')}
                            formId={texts.form.id}
                            title={texts.form.title}
                            />
                    )}
                </div>
                {popup && <Survey calculated={calculated} {...survey} />}
                <Footer />
                {data.metrika.id && (
                <YMInitializer
                    accounts={[data.metrika.id]}
                    options={data.metrika.options}
                    />
                )}
                <script src={shareScript} />
                <script src="https://yastatic.net/s3/vda/i/simplebar/5.0.3/simplebar.min.js" />
            </CommonDataContextProvider>
        </ConfigContextProvider>
    );
}
