import { classnames } from '@bem-react/classnames';
import React, {
    ChangeEvent,
    MouseEvent,
    UIEvent,
    useEffect,
    useReducer,
    useState
} from 'react';

import { useCommonData } from 'client/common/hooks';

import cn from 'utils/cn';
import { reachGoal } from 'utils/metrika';

import { Hint } from 'client/platforms/statvalue/components/hint';
import { Button, Input } from 'client/platforms/statvalue/components/lego';

import {
    getInitialState,
    normalizeNumber,
    reducer
} from './reducer';
import {
    ActionType,
    IColumn,
    ISegment,
    ISegmentsProps,
    ISegmentsState
} from './types';

import { Table } from './table';

import './index.css';

export const b = cn('segments');

export const alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
const layoutWidth = 1126;
const horizontalPaddings = 65;

export const getFixedLeftColumns = (titles: Record<string, string>): IColumn[] => ([
    {
        id: 'segment',
        title: titles.segment,
        width: 64,
        paddings: 46,
        getValue: (_, index) => alphabet[index]
    },
    {
        id: 'size',
        title: titles.size,
        width: 70,
        paddings: 10,
        editable: true,
        suffix: '%'
    },
    {
        id: 'cost',
        title: titles.cost,
        width: 90,
        paddings: 10,
        editable: true
    },
    {
        id: 'conversions',
        title: titles.conversions,
        width: 80,
        paddings: 10,
        editable: true
    },
    {
        id: 'clicks',
        title: titles.clicks,
        width: 80,
        paddings: 10,
        editable: true
    }
]);

export const getScrollingColumns = (titles: Record<string, string>): IColumn[] => ([
    {
        id: 'cpa',
        title: titles.cpa
    },
    {
        id: 'costRelative',
        title: titles.costRelative,
        suffix: '%'
    },
    {
        id: 'conversionsRelative',
        title: titles.conversionsRelative,
        suffix: '%'
    },
    {
        id: 'clicksRelative',
        title: titles.clicksRelative,
        suffix: '%'
    },
    {
        id: 'cpaRelative',
        title: titles.cpaRelative,
        suffix: '%'
    },
    {
        id: 'costErrorRelative',
        title: titles.costErrorRelative,
        suffix: '%'
    },
    {
        id: 'conversionsErrorRelative',
        title: titles.conversionsErrorRelative,
        suffix: '%'
    },
    {
        id: 'profit',
        title: titles.profit
    },
    {
        id: 'profitRelative',
        title: titles.profitRelative,
        suffix: '%'
    },
    {
        id: 'profitErrorRelative',
        title: titles.profitErrorRelative,
        suffix: '%'
    },
    {
        id: 'statValue',
        hintText: titles.statValueHint,
        title: titles.statValue,
        hideOnA: true
    },
    {
        id: 'pValue',
        title: titles.pValue,
        hideOnA: true,
        suffix: '%'
    }
]);

export const getFixedRightColumns = (titles: Record<string, string>): IColumn[] => ([
    {
        id: 'shipIt',
        title: titles.shipIt,
        width: 100,
        paddings: 20,
        hideOnA: true,
        getValue: segment => {
            switch (segment.shipIt) {
                case 'YES':
                    return titles.shipItYesValue;

                case 'NO':
                    return titles.shipItNoValue;

                case 'UNDEF':
                    return titles.shipItUndefValue;

                default:
                    return titles.shipItDefaultValue;
            }
        },
        getMods: segment => ({
            type: segment.shipIt
                ? segment.shipIt.toLowerCase()
                : false
        }),
        getTitle: (segment, index) => {
            const name = alphabet[index];
            const { pValue, shipIt } = segment;

            switch (shipIt) {
                case 'YES':
                    return titles.shipItYes
                        .replace('$name', name)
                        .replace('$pValue', pValue || 'undefined');

                case 'NO':
                    return titles.shipItNo
                        .replace('$name', name)
                        .replace('$pValue', pValue || 'undefined');

                case 'UNDEF':
                    return titles.shipItUndef
                        .replace('$name', name)
                        .replace('$pValue', pValue || 'undefined');

                default:
                    return titles.shipItDefault
                        .replace('$name', name)
                        .replace('$pValue', pValue || 'undefined');
            }
        }
    }
]);

function isAllFieldsFilled(state: ISegmentsState) {
    const isAllSegmentsFilled = state.segments.every(segment =>
        segment.size && segment.cost && segment.conversions && segment.clicks
    );

    return (
        Boolean(state.pValueTarget) &&
        Boolean(state.k) &&
        isAllSegmentsFilled
    );
}

function escapeCSVValue(value: string) {
    return value.includes(',')
        ? `"${value}"`
        : value;
}

function buildCSVBlob(segments: ISegment[], columns: IColumn[]) {
    const titles = columns.map(column => escapeCSVValue(column.title));
    const rows = segments.map((segment, segmentIndex) =>
        columns.map(column => {
            const text = column.getValue
                ? column.getValue(segment, segmentIndex)
                : segment[column.id];

            const isEmpty = typeof text === 'undefined' || text === '';
            const isHidden = column.hideOnA && segmentIndex === 0;

            if (isEmpty || isHidden) {
                return '—';
            }

            const content = column.suffix
                ? String(text) + column.suffix
                : String(text);

            return escapeCSVValue(content);
        })
    );

    const content = [titles, ...rows]
        .map(row => row.join(','))
        .join('\n');

    return new Blob(['\uFEFF', content], { type: 'text/csv;charset=utf-8' });
}

export function Segments({ calculator, className = '', onCalculate }: ISegmentsProps) {
    const fixedLeftColumns = getFixedLeftColumns(calculator.columns);
    const scrollingColumns = getScrollingColumns(calculator.columns);
    const fixedRightColumns = getFixedRightColumns(calculator.columns);
    const columns = [...fixedLeftColumns, ...scrollingColumns, ...fixedRightColumns];

    const { secretKey } = useCommonData();
    const [state, dispatch] = useReducer(reducer, getInitialState(calculator));

    const [hasLeftward, setHasLeftward] = useState(false);
    const [hasRightward, setHasRightward] = useState(true);
    const [isLoading, setIsLoading] = useState(false);
    const [csv, setCSV] = useState('');

    function handlePValueTargetChange({ target: { value } }: ChangeEvent<HTMLInputElement>) {
        dispatch({ type: ActionType.SetPValueTarget, payload: value });
    }

    function handleKChange({ target: { value } }: ChangeEvent<HTMLInputElement>) {
        dispatch({ type: ActionType.SetK, payload: value });
    }

    function handleSegmentAdd() {
        if (state.segments.length < alphabet.length) {
            dispatch({ type: ActionType.AddSegment });
        }
    }

    function handleSubmit() {
        const k = normalizeNumber(state.k);
        const pValueTarget = normalizeNumber(state.pValueTarget);
        const segments = state.segments.map(segment => ({
            size: normalizeNumber(segment.size),
            cost: normalizeNumber(segment.cost),
            conversions: normalizeNumber(segment.conversions),
            clicks: normalizeNumber(segment.clicks)
        }));

        setIsLoading(true);

        fetch('/vda/statvalue', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-Token': secretKey
            },
            credentials: 'same-origin',
            body: JSON.stringify({ k, pValueTarget, segments })
        })
            .then(response => response.json())
            .then(payload => {
                dispatch({ type: ActionType.SetResults, payload });
                setIsLoading(false);
                onCalculate();
            });

        reachGoal('click-result-button', { k, pValueTarget, segments });
    }

    function handleScroll(event: UIEvent<HTMLDivElement>) {
        const { scrollLeft, offsetWidth, scrollWidth } = event.target as HTMLDivElement;

        setHasLeftward(scrollLeft > 0);
        setHasRightward(scrollLeft + offsetWidth < scrollWidth);
    }

    function handleDownloadExcelClick() {
        reachGoal('click-download-exel');
    }

    function handleDownloadCSVClick(event: MouseEvent<HTMLAnchorElement>) {
        reachGoal('csv-download');

        // В IE11 не поддерживается атрибут download
        const anchor = document.createElement('a');
        const isDownloadAttributeSupported = typeof anchor.download !== 'undefined';

        if (!isDownloadAttributeSupported && navigator.msSaveBlob) {
            event.preventDefault();

            const blob = buildCSVBlob(state.segments, columns);

            navigator.msSaveBlob(blob, 'result.csv');
        }
    }

    function handleMouseOver(event: MouseEvent<HTMLDivElement>) {
        const { target } = event;
        const parent = (target as HTMLElement).closest('.segments__row');

        const suitableIndex = parent && Number(parent.getAttribute('data-index'));

        if (suitableIndex && suitableIndex > 0) {
            dispatch({ type: ActionType.SetHoveredSegment, payload: suitableIndex });
        }
    }

    function createRemoveSegmentHandler(payload: number) {
        return function (event: MouseEvent<HTMLDivElement>) {
            const { target } = event;

            if (!(target as HTMLDivElement).classList.contains('segments__remove-icon_visible')) {
                return;
            }

            dispatch({ type: ActionType.RemoveSegment, payload });
        };
    }

    useEffect(() => {
        if (state.hasResults) {
            const csvBlob = buildCSVBlob(state.segments, columns);

            setCSV(URL.createObjectURL(csvBlob));
        }
    }, [state.hasResults, state.segments, columns]);

    const fixedColumnsWidth = [...fixedLeftColumns, ...fixedRightColumns]
        .reduce((sum, column) => {
            const width = column.width || 0;
            const paddings = column.paddings || 0;

            return sum + width + paddings;
        }, 0);
    const scrollingTableWidth = layoutWidth - horizontalPaddings - fixedColumnsWidth;

    return (
        <div className={classnames(b(), className)} >
            <div id="calculator" className={b('header')}>
                <div className={b('header-left')}>
                    <div className={b('p-value')}>
                        <Hint className={b('hint')} text={calculator.pValueHint} />
                        <div
                            className={b('p-value-text')}
                            dangerouslySetInnerHTML={{ __html: calculator.pValueText }}
                            />
                        <Input
                            size="m"
                            theme="normal"
                            className={b('p-value-input')}
                            value={state.pValueTarget}
                            onChange={handlePValueTargetChange}
                            />
                        <span className={b('cell-suffix')}>%</span>
                    </div>
                    <div className={b('k')}>
                        <Hint className={b('hint')} text={calculator.kHint} />
                        <div
                            className={b('k-text')}
                            dangerouslySetInnerHTML={{ __html: calculator.kText }}
                            />
                        <Input
                            size="m"
                            theme="normal"
                            className={b('k-input')}
                            value={state.k}
                            onChange={handleKChange}
                            />
                    </div>
                </div>
                <div className={b('header-right')}>
                    <Button
                        size="l"
                        target="_blank"
                        theme="normal"
                        type="link"
                        url={calculator.excelLink}
                        onClick={handleDownloadExcelClick}
                        >
                        <div className={b('excel-icon')} /> {calculator.excelLinkText}
                    </Button>
                </div>
            </div>
            <div className={b('table-wrapper', { leftward: hasLeftward, rightward: hasRightward })} onMouseOver={handleMouseOver}>
                <Table
                    position="left"
                    columns={fixedLeftColumns}
                    segments={state.segments}
                    dispatch={dispatch}
                    />
                <div
                    data-simplebar
                    data-simplebar-auto-hide="false"
                    className={b('table-scroller')}
                    style={{ width: scrollingTableWidth }}
                    onScroll={handleScroll}
                    >
                    <Table
                        columns={scrollingColumns}
                        segments={state.segments}
                        dispatch={dispatch}
                        />
                </div>
                <Table
                    position="right"
                    columns={fixedRightColumns}
                    segments={state.segments}
                    dispatch={dispatch}
                    />
                <table className={b('remove-icons')}>
                    <thead>
                        <tr>
                            <th className={b('remove-cell')} />
                        </tr>
                    </thead>
                    <tbody>
                        {state.segments.map((_, segmentIndex) => (
                            <tr key={alphabet[segmentIndex]}>
                                <td className={b('remove-cell')}>
                                    {state.segments.length > 2 && segmentIndex !== 0 && (
                                        <div
                                            className={b('remove-icon', { visible: state.hoveredSegment === segmentIndex })}
                                            onClick={createRemoveSegmentHandler(segmentIndex)}
                                            />
                                    )}
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>
            {state.segments.length < alphabet.length && (
                <div className={b('add-segment')} onClick={handleSegmentAdd}>
                    <div className={b('add-segment-icon')} />
                </div>
            )}
            <div className={b('calculate')}>
                <Button
                    size="l"
                    theme="action"
                    className={b('button', { pristine: state.pristine })}
                    disabled={state.pristine || isLoading || !isAllFieldsFilled(state)}
                    onClick={handleSubmit}
                    >
                    {calculator.calculateButtonText}
                </Button>
                {csv && (
                    <a
                        download="result.csv"
                        className={b('csv')}
                        href={csv}
                        onClick={handleDownloadCSVClick}
                        >
                        {calculator.csvLinkText}
                    </a>
                )}
            </div>
        </div>
    );
}
