import { IStatvalueCalculator } from 'client/common/types';

import { ActionType, ISegmentsState, TAction } from './types';

const emptySegment = { size: '50', cost: '', conversions: '', clicks: '' };

function isNumber(value: string) {
    return value.endsWith('e')
        ? !isNaN(Number(`${value}0`))
        : !isNaN(Number(value));
}

function preprocessNumber(value: string) {
    return value
        // Убираем все пробелы от форматирования
        .replace(/\s/g, '')
        // Заменяем опечатки в разделителе
        .replace(/[,юб]/g, '.')
        // Заменяем опечатки в экспоненциальной записи
        .replace(/[уtе]/g, 'e')
        // Исправлем ведущую точку в начале
        .replace(/^\./, '0.')
        // Заменяем ведущие нули на один
        .replace(/^0+/, '0')
        // Убираем ведущий ноль, если он не имеет смысла
        .replace(/^0+([^0.e])/, '$1');
}

export function prettifyNumber(value: string = '') {
    const [integerPart, fractionalPart] = value.split('.');

    if (!integerPart) {
        return '';
    }

    let prettifiedIntegerPart;

    // Целую часть (до экспоненциальной записи) разбиваем на группы по три разряда
    if (integerPart.includes('e')) {
        const [beforeExponential, exponential] = integerPart.split('e');
        const prettifiedBeforeExponential = beforeExponential
            .replace(/\B(?=(\d{3})+(?!\d))/g, ' ');

        prettifiedIntegerPart = `${prettifiedBeforeExponential}e${exponential}`;
    } else {
        prettifiedIntegerPart = integerPart
            .replace(/\B(?=(\d{3})+(?!\d))/g, ' ');
    }

    if (typeof fractionalPart === 'undefined') {
        return prettifiedIntegerPart;
    }

    return `${prettifiedIntegerPart}.${fractionalPart}`;
}

export function normalizeNumber(value: string) {
    return parseFloat(value.replace(/\s/g, ''));
}

function changeNumber(prevValue: string | undefined, nextValue: string) {
    nextValue = preprocessNumber(nextValue);

    if (!isNumber(nextValue)) {
        return prevValue || '';
    }

    return prettifyNumber(nextValue);
}

export function getInitialState(options: IStatvalueCalculator): ISegmentsState {
    return {
        pValueTarget: options.pValueDefault,
        k: options.kDefault,
        segments: [
            { ...emptySegment },
            { ...emptySegment }
        ],
        hasResults: false,
        pristine: false,
        hoveredSegment: 0
    };
}

export function reducer(state: ISegmentsState, action: TAction): ISegmentsState {
    switch (action.type) {
        case ActionType.AddSegment: {
            return {
                ...state,
                pristine: false,
                segments: [...state.segments, { ...emptySegment }]
            };
        }

        case ActionType.SetResults: {
            return {
                ...state,
                pristine: true,
                segments: state.segments.map((segment, index) => ({
                    ...segment,
                    ...action.payload[index]
                })),
                hasResults: true
            };
        }

        case ActionType.SetSegmentField: {
            const prevValue = state.segments[action.payload.index][action.payload.id];
            const nextValue = action.payload.value;

            return {
                ...state,
                pristine: false,
                segments: [
                    ...state.segments.slice(0, action.payload.index),
                    {
                        ...state.segments[action.payload.index],
                        [action.payload.id]: changeNumber(prevValue, nextValue)
                    },
                    ...state.segments.slice(action.payload.index + 1)
                ]
            };
        }

        case ActionType.SetPValueTarget: {
            const prevValue = state.pValueTarget;
            const nextValue = action.payload;

            return {
                ...state,
                pristine: false,
                pValueTarget: changeNumber(prevValue, nextValue)
            };
        }

        case ActionType.SetK: {
            const prevValue = state.k;
            const nextValue = action.payload;

            return {
                ...state,
                pristine: false,
                k: changeNumber(prevValue, nextValue)
            };
        }

        case ActionType.RemoveSegment: {
            const segments = [...state.segments];

            segments.splice(action.payload, 1);

            return {
                ...state,
                pristine: false,
                segments
            };
        }

        case ActionType.SetHoveredSegment: {
            const hoveredSegment = action.payload;

            return {
                ...state,
                hoveredSegment
            };
        }

        default:
            return state;
    }
}
