/* eslint complexity: 0 */

import React, {
    ChangeEvent,
    Dispatch,
    Fragment
} from 'react';

import { Hint } from 'client/platforms/statvalue/components/hint';
import { Input } from 'client/platforms/statvalue/components/lego';

import { alphabet, b } from '../index';
import { prettifyNumber } from '../reducer';
import {
    ActionType,
    IColumn,
    ISegment,
    TAction
} from '../types';

interface ICellProps {
    columns: IColumn[];
    columnIndex: number;
    dispatch: Dispatch<TAction>;
    segment: ISegment;
    segmentIndex: number;
}

interface IHeadingCellProps {
    columns: IColumn[];
    columnIndex: number;
}

function HeadingCell({ columns, columnIndex }: IHeadingCellProps) {
    const column = columns[columnIndex];
    const className = b('cell', {
        editable: column.editable,
        'fixed-width': Boolean(column.width),
        hint: Boolean(column.hintText),
        heading: true
    });

    return (
        <th key={column.title} className={className}>
            {column.hintText && <Hint text={column.hintText} />}
            <div
                className={b('cell-content')}
                dangerouslySetInnerHTML={{ __html: column.title }}
                style={{ width: column.width }}
                />
        </th>
    );
}

function Cell(props: ICellProps) {
    const {
        columns,
        columnIndex,
        dispatch,
        segment,
        segmentIndex
    } = props;

    const column = columns[columnIndex];

    const textContent = column.getValue
        ? column.getValue(segment, segmentIndex)
        : prettifyNumber(segment[column.id]);
    const title = column.getTitle && column.getTitle(segment, segmentIndex);

    const shouldShowContent = !column.hideOnA || segmentIndex > 0;
    const mods = shouldShowContent && column.getMods
        ? column.getMods(segment, segmentIndex)
        : {};
    const className = b('cell', {
        editable: column.editable,
        'fixed-width': Boolean(column.width),
        ...mods
    });

    let content = null;

    if (column.editable) {
        content = (
            <Input
                size="m"
                theme="normal"
                value={textContent as string}
                onChange={createFieldChangeHandler(segmentIndex, column.id)}
                />
        );
    } else if (shouldShowContent) {
        content = textContent;
    }

    const hasContent = typeof content !== 'undefined' && content !== '';

    function createFieldChangeHandler(index: number, id: string) {
        return ({ target: { value } }: ChangeEvent<HTMLInputElement>) => {
            dispatch({
                type: ActionType.SetSegmentField,
                payload: { index, value, id }
            });
        };
    }

    return (
        <td key={column.title} className={className} title={title}>
            <div className={b('cell-content')} style={{ width: column.width }}>
                {hasContent
                    ? (
                        <Fragment>
                            {content}
                            {column.suffix && shouldShowContent && (
                                <span className={b('cell-suffix')}>{column.suffix}</span>
                            )}
                        </Fragment>
                    )
                    : '—'
                }
            </div>
        </td>
    );
}

interface ITableProps {
    columns: IColumn[];
    position?: 'left' | 'right';
    segments: ISegment[];
    dispatch: Dispatch<TAction>;
}

export function Table({ columns, position, segments, dispatch }: ITableProps) {
    return (
        <table className={b('table', { position })}>
            <thead>
                <tr className={b('row', { heading: true })}>
                    {columns.map((column, columnIndex) => (
                        <HeadingCell
                            key={column.title}
                            columns={columns}
                            columnIndex={columnIndex}
                            />
                    ))}
                </tr>
            </thead>
            <tbody>
                {segments.map((segment, segmentIndex) => (
                    <tr key={alphabet[segmentIndex]} className={b('row', { content: true })} data-index={segmentIndex}>
                        {columns.map((column, columnIndex) => (
                            <Cell
                                key={column.title}
                                columns={columns}
                                columnIndex={columnIndex}
                                dispatch={dispatch}
                                segment={segment}
                                segmentIndex={segmentIndex}
                                />
                        ))}
                    </tr>
                ))}
            </tbody>
        </table>
    );
}
