import React, { useEffect, useState } from 'react';
import debounce from 'utils/debounce';

import HtmlContent from './html-content';

interface IHtmlContentWrapperProps {
    className?: string;
    content?: string;
    tag?: string;
    theme?: string;
    enabledTouchAdaptiveIframe?: boolean;
}

interface IframeParams {
    scale: number;
    width: string;
    height: string;
}

interface IframeStyles {
    parentStyles: string;
    iframeStyles: string;
}

function HtmlContentWrapper(props: IHtmlContentWrapperProps) {
    const [content, setContent] = useState<string | undefined>();
    const [contentNode, setContentNode] = useState<HTMLElement | undefined>();

    const scaleIframes = () => setContent(getContentWithScaledIframes(props, contentNode));
    const onResizeDebounced = debounce(scaleIframes, 300, false);

    useEffect(() => {
        window.addEventListener('orientationchange', onResizeDebounced);

        return () => window.removeEventListener('orientationchange', onResizeDebounced);
    }, [onResizeDebounced]);
    useEffect(scaleIframes, [contentNode]);

    if (!content) {
        return null;
    }

    return (
        <HtmlContent
            {...props}
            content={content}
            onRender={setContentNode}
            />
    );
}

function getContentWithScaledIframes(
    props: IHtmlContentWrapperProps,
    contentNode: HTMLElement | undefined
): string | undefined {
    const { content: initialContent, enabledTouchAdaptiveIframe } = props;

    // Сначала контент отображаем как есть, чтобы узнать реальные размеры ифрейма
    if (!initialContent || !contentNode) {
        return initialContent;
    }

    const iframes = contentNode.getElementsByTagName('iframe');

    // Если ифреймов нет, не парсить.
    if (!iframes || iframes.length === 0) {
        return initialContent;
    }

    const params = Array.from(iframes).map(iframe => getIframeStyles(iframe, contentNode, enabledTouchAdaptiveIframe));
    let index = 0;

    return initialContent.replace(/<iframe.*?><\/iframe>/g, (...match) => {
        const styles = params[index];

        index = index + 1;
        const parsedIframe = `<iframe style="${styles.iframeStyles}" $1></iframe>`;

        return match[0].replace(
            /<iframe(.*?)><\/iframe>/,
            `<span class="html-content__iframe-wrapper" style="${styles.parentStyles}">${parsedIframe}</span>`
        );
    });
}

function getIframeStyles(iframe: HTMLIFrameElement, contentNode: HTMLElement, enabledTouchAdaptiveIframe?: boolean): IframeStyles {
    if (enabledTouchAdaptiveIframe) {
        return {
            parentStyles: '',
            iframeStyles: isIframeWideThenContainer(iframe, contentNode) ? 'width: 100%' : ''
        };
    }

    const scaledParams = getScaledIframeParams(iframe, contentNode);

    return {
        parentStyles: `width: ${scaledParams.width}; height: ${scaledParams.height}`,
        iframeStyles: `transform-origin: left top; margin: 0px; transform: scale(${scaledParams.scale})`
    };
}

function getScaledIframeParams(iframe: HTMLIFrameElement, contentNode: HTMLElement): IframeParams {
    const contentNodeWidth = contentNode.offsetWidth;
    const iframeWidth = iframe.offsetWidth;

    if (!isIframeWideThenContainer(iframe, contentNode)) {
        return {
            scale: 1,
            width: 'auto',
            height: 'auto'
        };
    }

    // iframe шире чем viewport, вычисляем коэффициент масштабирования
    const scale = contentNodeWidth / iframeWidth;

    return {
        scale,
        width: `${contentNodeWidth}px`,
        height: `${(iframe.offsetHeight * scale)}px`
    };
}

function isIframeWideThenContainer(iframe: HTMLIFrameElement, contentNode: HTMLElement): boolean {
    return contentNode.offsetWidth < iframe.offsetWidth;
}

export default HtmlContentWrapper;
