import { handleActions } from 'redux-actions';

import { CertificateType } from 'client/common/types';

import {
    ActionTypes,
    IAddMultiFilterAction,
    IAgenciesState,
    IChangeSingleFilterAction,
    IFailLoadingAction,
    IFinishLoadingAction,
    IRemoveMultiFilterAction,
    IReplaceMultiFilterAction,
    IResetFiltersAction,
    ISetBudgetAction,
    ISetBudgetOrderAction,
    IStartLoadingAction,
    IStartLoadingFromScratchAction,
    TAgenciesPayload
} from './types';

const initialState: IAgenciesState = {
    filters: {
        city: [],
        certificate: [],
        service: ['full'],
        budget: undefined,
        budgetOrder: undefined
    },
    rawFilters: {
        certificates: [],
        cities: [],
        countries: [],
        services: []
    },
    lists: {
        main: {
            agencies: [],
            hasMore: false,
            seed: ''
        },
        offices: {
            agencies: [],
            hasMore: false,
            seed: ''
        },
        partners: {
            agencies: [],
            hasMore: false,
            seed: ''
        }
    },
    isError: false
};

function normalizeState(state: IAgenciesState) {
    if (state.filters.certificate.indexOf(CertificateType.Direct) === -1) {
        state.filters.budget = undefined;
        state.filters.budgetOrder = undefined;
    }

    return state;
}

function filterServices(state: IAgenciesState) {
    const checkedCertificate = state.filters.certificate;

    if (!state.filters.service || !state.filters.service.length) {
        return state;
    }

    const readyService = state.rawFilters.services.filter(item => {
        let result = false;

        for (let i = 0; i < checkedCertificate.length; i++) {
            result = item.certificates.includes(checkedCertificate[i]) &&
            state.filters.service.includes(item.code);

            if (!result) {
                return result;
            }
        }

        return result;
    });

    state.filters.service = readyService.map(item => item.code);

    return state;
}

function addMultiFilter(state: IAgenciesState, action: IAddMultiFilterAction) {
    const { filterId, optionId } = action.payload!;
    const { filters } = state;
    const copyFilters = filters[filterId] ? filters[filterId] : [];

    const checked = optionId === 'all' ? [optionId] : [...copyFilters, optionId];

    const newState = {
        ...state,
        filters: {
            ...filters,
            [filterId]: checked
        }
    };

    return filterServices(newState);
}

function replaceMultiFilter(state: IAgenciesState, action: IReplaceMultiFilterAction) {
    const { filterId, index, optionId } = action.payload!;
    const { filters } = state;

    if (optionId === 'all') {
        const newState = {
            ...state,
            filters: {
                ...filters,
                [filterId]: [optionId]
            }
        };

        return filterServices(newState);
    }

    const checked = [...filters[filterId]];

    checked[index] = optionId;

    const newState = {
        ...state,
        filters: {
            ...filters,
            [filterId]: checked
        }
    };

    return filterServices(normalizeState(newState));
}

function removeMultiFilter(state: IAgenciesState, action: IRemoveMultiFilterAction) {
    const { filterId, optionId } = action.payload!;
    const { filters } = state;

    const checked = filters[filterId].filter(value => value !== optionId);

    const newState = {
        ...state,
        filters: {
            ...filters,
            [filterId]: checked
        }
    };

    return filterServices(normalizeState(newState));
}

function changeSingleFilter(state: IAgenciesState, action: IChangeSingleFilterAction) {
    const { filterId, value, relatedFilter } = action.payload!;
    const { filters } = state;

    const { filterId: relatedFilterId, value: relatedFilterValue } = relatedFilter;

    const newState = {
        ...state,
        filters: {
            ...filters,
            [filterId]: value,
            [relatedFilterId]: relatedFilterValue
        }
    };

    return filterServices(newState);
}

function resetFilters(state: IAgenciesState, action: IResetFiltersAction) {
    const { filters } = action.payload!;

    return {
        ...state,
        filters: {
            ...filters,
            budget: undefined,
            budgetOrder: undefined
        }
    };
}

function setBudget(state: IAgenciesState, action: ISetBudgetAction) {
    const { filters } = state;
    const { budget } = action.payload!;

    return {
        ...state,
        filters: {
            ...filters,
            budget
        }
    };
}

function setBudgetOrder(state: IAgenciesState, action: ISetBudgetOrderAction) {
    const { filters } = state;
    const { budgetOrder } = action.payload!;

    return {
        ...state,
        filters: {
            ...filters,
            budgetOrder
        }
    };
}

function startLoading(state: IAgenciesState, action: IStartLoadingAction) {
    const { group } = action.payload!;
    const { lists } = state;
    const groupList = lists[group];

    return {
        ...state,
        lists: {
            ...lists,
            [group]: {
                ...groupList,
                isLoading: true
            }
        },
        isError: false
    };
}

function startLoadingFromScratch(state: IAgenciesState, action: IStartLoadingFromScratchAction) {
    const { group } = action.payload!;
    const { lists } = state;
    const groupList = lists[group];

    return {
        ...state,
        lists: {
            ...lists,
            [group]: {
                ...groupList,
                agencies: [],
                hasMore: true,
                isLoading: true
            }
        },
        isError: false
    };
}

function finishLoading(state: IAgenciesState, action: IFinishLoadingAction) {
    const { group, agencies, hasMore } = action.payload!;
    const { lists } = state;
    const groupList = lists[group];

    return {
        ...state,
        lists: {
            ...lists,
            [group]: {
                ...groupList,
                agencies: [...groupList.agencies, ...agencies],
                hasMore,
                isLoading: false
            }
        }
    };
}

function failLoading(state: IAgenciesState, action: IFailLoadingAction) {
    const { group } = action.payload!;
    const { lists } = state;
    const groupList = lists[group];

    return {
        ...state,
        lists: {
            ...lists,
            [group]: {
                ...groupList,
                isLoading: false
            }
        },
        isError: true
    };
}

export default handleActions<IAgenciesState, TAgenciesPayload>({
    [ActionTypes.ADD_MULTI_FILTER]: addMultiFilter,
    [ActionTypes.REPLACE_MULTI_FILTER]: replaceMultiFilter,
    [ActionTypes.REMOVE_MULTI_FILTER]: removeMultiFilter,
    [ActionTypes.CHANGE_SINGLE_FILTER]: changeSingleFilter,
    [ActionTypes.RESET_FILTERS]: resetFilters,
    [ActionTypes.SET_BUDGET]: setBudget,
    [ActionTypes.SET_BUDGET_ORDER]: setBudgetOrder,
    [ActionTypes.START_LOADING]: startLoading,
    [ActionTypes.START_LOADING_FROM_SCRATCH]: startLoadingFromScratch,
    [ActionTypes.FINISH_LOADING]: finishLoading,
    [ActionTypes.FAIL_LOADING]: failLoading
}, initialState);
