import { handleActions } from 'redux-actions';

import { LoadingState } from 'client/common/types/common';

import {
    ActionTypes,
    AgencyFormMode,
    IAddFormOfficeAction,
    IAgencyPayload,
    IAgencyState,
    IChangeFormDataAction,
    IChangeFormOfficeAction,
    IRemoveFormOfficeAction,
    ISelectOfficeAction,
    IShowOrHideProductsCardAction,
    IStartEditingAction
} from './types';

const initialState: IAgencyState = {
    form: {
        mode: AgencyFormMode.hidden,
        data: null,
        sendingState: LoadingState.none
    },
    offices: {
        selectedId: null
    },
    showProductsCard: {
        market: false,
        direct: false,
        metrika: false,
        toloka: false,
        dialogs: false
    }
};

function showOrHideProductsCard(state: IAgencyState, action: IShowOrHideProductsCardAction): IAgencyState {
    const newState = { ...state };

    if (action.payload?.certificateName) {
        newState.showProductsCard = newState.showProductsCard ? { ...newState.showProductsCard } : {
            market: false,
            direct: false,
            metrika: false,
            toloka: false,
            dialogs: false
        };

        newState.showProductsCard[action.payload.certificateName] = action.payload?.showProductsCard;
    }

    return newState;
}

function startEditing(state: IAgencyState, action: IStartEditingAction): IAgencyState {
    const { initialData = null } = action.payload || {};

    return {
        ...state,
        form: {
            mode: AgencyFormMode.editing,
            data: initialData,
            sendingState: LoadingState.none
        }
    };
}

function cancelEditing(state: IAgencyState): IAgencyState {
    return {
        ...state,
        form: {
            mode: AgencyFormMode.hidden,
            data: null,
            sendingState: LoadingState.none
        }
    };
}

function changeFormData(state: IAgencyState, action: IChangeFormDataAction): IAgencyState {
    const { form } = state;

    if (!form.data) {
        return state;
    }

    return {
        ...state,
        form: {
            ...form,
            data: {
                ...form.data,
                ...action.payload?.changes
            }
        }
    };
}

function changeFormOffice(state: IAgencyState, action: IChangeFormOfficeAction): IAgencyState {
    const { form } = state;

    if (!form.data || !action.payload) {
        return state;
    }

    const { index, changes } = action.payload;
    const offices = form.data.offices.slice();

    offices[index] = {
        ...offices[index],
        ...changes
    };

    return {
        ...state,
        form: {
            ...form,
            data: {
                ...form.data,
                offices
            }
        }
    };
}

function addFormOffice(state: IAgencyState, action: IAddFormOfficeAction): IAgencyState {
    const { form } = state;

    if (!form.data || !action.payload) {
        return state;
    }

    const { office } = action.payload;
    const offices = [
        ...form.data.offices,
        office
    ];

    return {
        ...state,
        form: {
            ...form,
            data: {
                ...form.data,
                offices
            }
        }
    };
}

function removeFormOffice(state: IAgencyState, action: IRemoveFormOfficeAction): IAgencyState {
    const { form } = state;

    if (!form.data || !action.payload) {
        return state;
    }

    const { index } = action.payload;
    const offices = form.data.offices.slice();

    offices.splice(index, 1);

    return {
        ...state,
        form: {
            ...form,
            data: {
                ...form.data,
                offices
            }
        }
    };
}

function showFormPreview(state: IAgencyState): IAgencyState {
    return {
        ...state,
        form: {
            ...state.form,
            mode: AgencyFormMode.preview
        }
    };
}

function hideFormPreview(state: IAgencyState): IAgencyState {
    return {
        ...state,
        form: {
            ...state.form,
            mode: AgencyFormMode.editing
        }
    };
}

function sendingStart(state: IAgencyState): IAgencyState {
    return {
        ...state,
        form: {
            ...state.form,
            sendingState: LoadingState.inProgress
        }
    };
}

function sendingFail(state: IAgencyState): IAgencyState {
    return {
        ...state,
        form: {
            ...state.form,
            sendingState: LoadingState.fail
        }
    };
}

function sendingSuccess(state: IAgencyState): IAgencyState {
    return {
        ...state,
        form: {
            ...state.form,
            mode: AgencyFormMode.hidden,
            data: null,
            sendingState: LoadingState.success
        }
    };
}

function selectOffice(state: IAgencyState, action: ISelectOfficeAction): IAgencyState {
    return {
        ...state,
        offices: {
            selectedId: action.payload!.officeId
        }
    };
}

export default handleActions<IAgencyState, IAgencyPayload>({
    [ActionTypes.START_EDITING]: startEditing,
    [ActionTypes.CANCEL_EDITING]: cancelEditing,
    [ActionTypes.CHANGE_FORM_DATA]: changeFormData,
    [ActionTypes.CHANGE_FORM_OFFICE]: changeFormOffice,
    [ActionTypes.ADD_FORM_OFFICE]: addFormOffice,
    [ActionTypes.REMOVE_FORM_OFFICE]: removeFormOffice,
    [ActionTypes.SHOW_FORM_PREVIEW]: showFormPreview,
    [ActionTypes.HIDE_FORM_PREVIEW]: hideFormPreview,
    [ActionTypes.SENDING_START]: sendingStart,
    [ActionTypes.SENDING_FAIL]: sendingFail,
    [ActionTypes.SENDING_SUCCESS]: sendingSuccess,
    [ActionTypes.SELECT_OFFICE]: selectOffice,
    [ActionTypes.SHOW_OR_HIDE_PRODUCT_CARD]: showOrHideProductsCard
}, initialState);
