import i18n from 'utils/i18n';

const DATE_REGEXP = /(\d?\d)\.(\d?\d)\.(\d\d\d\d)/;

export function getDate(dateString: string) {
    const match = dateString && dateString.match(DATE_REGEXP);
    const dateParts = match && match.slice(1, 4).map(Number);

    const month = dateParts && dateParts[1] - 1;
    const [day,, year] = dateParts as number[];

    return dateParts && new Date(year, month as number, day);
}

type TPatternsKeys =
    | 'short'
    | 'long'
    | 'full'
    | 'datetime';

interface IPatternsElement {
    en: string;
    'zh-Hans'?: string;
    default: string;
}

const patterns: Record<TPatternsKeys, IPatternsElement> = {
    short: {
        en: 'MMM D',
        default: 'D MMM'
    },
    long: {
        en: 'MMMM D',
        default: 'D MMMM'
    },
    full: {
        en: 'MMMM D, YYYY',
        'zh-Hans': 'YYYY年MMM月D日',
        default: 'D MMMM YYYY'
    },
    datetime: {
        en: 'MMMM D, YYYY HH:MM',
        default: 'D MMMM YYYY, HH:MM'
    }
};

const getMonthName = function (monthAlias: string, lang?: string) {
    // FIXME: keyset: i-tanker__months
    const monthName = i18n({
        keyset: 'months',
        key: monthAlias
    });

    if (lang !== 'en') {
        return monthName.toLowerCase();
    }

    return monthName;
};

type TComponentsKeys =
    | 'D'
    | 'MMM'
    | 'MMMM'
    | 'YYYY'
    | 'HH'
    | 'MM';

const components:
    Record<TComponentsKeys, (date: Date, lang?: string) => number | string> = {
        D(date: Date) {
            return date.getDate();
        },
        MMM(date: Date, lang?: keyof IPatternsElement) {
            const monthNum = date.getMonth() + 1;

            return getMonthName(`short${monthNum}`, lang) || monthNum;
        },
        MMMM(date: Date, lang?: keyof IPatternsElement) {
            const monthNum = date.getMonth() + 1;

            return getMonthName(`g${monthNum}`, lang);
        },
        YYYY(date: Date) {
            return date.getFullYear();
        },
        HH(date: Date) {
            return (`0${date.getHours()}`).slice(-2);
        },
        MM(date: Date) {
            return (`0${date.getMinutes()}`).slice(-2);
        }
    };

/**
 * Возвращает дату в человекопонятном виде по переданному формату
 * Если шаблон не передан, то дата вернется в формате D MMMM YYYY
 */
export function humanizeDate(
    date: Date | string | number,
    format: TPatternsKeys,
    lang?: string
): string {
    const pattern = getPattern(format, lang);

    date = createDate(date);

    return formatDate(date, pattern, lang);
}

function formatDate(date: Date, pattern: string, lang?: string) {
    return pattern.replace(/([\w]+)/g, (_str, formatSym: TComponentsKeys) => {
        return components[formatSym](date, lang) as string;
    });
}

function createDate(date: Date | string | number) {
    if (typeof date === 'string' || typeof date === 'number') {
        return new Date(date);
    }

    return date;
}

function getPattern(format: TPatternsKeys, lang?: string) {
    return patterns[format]
        ? ((lang && patterns[format][lang as keyof IPatternsElement]) || patterns[format].default)
        : '';
}
