'use strict';

const request = require('supertest');
const { expect } = require('chai');
const middlewareMock = require('./helper/middleware-mock');
const mockBlackbox = require('./helper/blackbox');
const nockTvm = require('./helper/tvm');
const config = require('yandex-cfg');
const nock = require('nock');
const url = require('url');
const mockery = require('mockery');

function middlewareMockBefore() {
    [
        'uatraits',
        'yandexuid',
        'secretkey',
        'render',
        'realBunker',
        'renderWithLocals',
        'secretkeyCheck'
    ]
        .forEach(middleware => middlewareMock[middleware]());

    mockery.enable({
        warnOnReplace: false,
        warnOnUnregistered: false,
        useCleanCache: true
    });
}

describe('Partners', () => {
    let app;
    const direct = {
        host: url.format({
            protocol: config.direct.protocol,
            hostname: config.direct.hostname,
            port: config.direct.port || null
        }),
        path: `/${config.direct.pathname}`
    };

    before(() => {
        nock(direct.host)
            .get(direct.path)
            .query({
                uid: '10001'
            })
            .times(Infinity)
            .reply(200, {
                role: 'client'
            });

        nock(direct.host)
            .get(direct.path)
            .query({
                uid: '10002'
            })
            .times(Infinity)
            .reply(200, {
                role: 'agency'
            });

        nock(direct.host)
            .get(direct.path)
            .query({
                uid: '10003'
            })
            .times(Infinity)
            .reply(200, {
                role: 'support'
            });

        nockTvm();
    });

    after(nock.cleanAll);

    afterEach(middlewareMock.integrationAfter);

    it('should set access to "forbidden" for unauthorized user', done => {
        mockBlackbox({ status: 'INVALID' });
        middlewareMockBefore();
        app = require('../server/app');

        request(app)
            .get('/adv/partners/analytics')
            .set('host', 'yandex.ru')
            .expect(200)
            .end((err, data) => {
                expect(data.body.access).to.equal('forbidden');
                done(err);
            });
    });

    it('should set access to "invalid" for not-agency', done => {
        mockBlackbox({
            status: 'VALID',
            raw: { uid: { value: '10001' } }
        });
        middlewareMockBefore();
        app = require('../server/app');

        request(app)
            .get('/adv/partners/analytics')
            .set('host', 'yandex.ru')
            .expect(200)
            .end((err, data) => {
                expect(data.body.role).to.equal('client');
                expect(data.body.access).to.equal('invalid');
                done(err);
            });
    });

    it('should set access to "allowed" for agency', done => {
        mockBlackbox({
            status: 'VALID',
            raw: { uid: { value: '10002' } }
        });
        middlewareMockBefore();
        app = require('../server/app');

        request(app)
            .get('/adv/partners')
            .set('host', 'yandex.ru')
            .expect(200)
            .end((err, data) => {
                expect(data.body.partners.role).to.equal('agency');
                expect(data.body.partners.access).to.equal('allowed');
                done(err);
            });
    });

    it('should set access to "allowed" for support', done => {
        mockBlackbox({
            status: 'VALID',
            raw: { uid: { value: '10003' } }
        });
        middlewareMockBefore();
        app = require('../server/app');

        request(app)
            .get('/adv/partners')
            .set('host', 'yandex.ru')
            .expect(200)
            .end((err, data) => {
                expect(data.body.partners.role).to.equal('support');
                expect(data.body.partners.access).to.equal('allowed');
                done(err);
            });
    });
});
