from django.contrib import admin

from modeltranslation.admin import TabbedTranslationAdmin, TranslationTabularInline
from moderation.models import ModeratedObject
from moderation.admin import ModerationAdmin, ModeratedObjectAdmin

from . import models, mixins
from .decorators import translate_list_display
from .utils import get_translation


class NameOnlyAdmin(TabbedTranslationAdmin):
    list_display = ('name',)


@translate_list_display(models.City)
@admin.register(models.City)
class CityAdmin(mixins.AdminPermissionMixin,
                TabbedTranslationAdmin):
    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor')
        }
    }

    roles_model = models.UserRole

    entity_field = None

    list_display = (
        'name_translation',
        'preposition_translation',
        'name_prepositional_translation',
        'geo_id',
        'country'
    )


@translate_list_display(models.Country)
@admin.register(models.Country)
class CountryAdmin(mixins.AdminPermissionMixin,
                   TabbedTranslationAdmin):
    permissions_dict = {
        'default': {
            'common': tuple()
        }
    }

    roles_model = models.UserRole

    entity_field = None

    list_display = (
        'name_translation',
        'preposition_translation',
        'name_prepositional_translation',
        'geo_id'
    )


@translate_list_display(models.Region)
@admin.register(models.Region)
class RegionAdmin(mixins.AdminPermissionMixin,
                  TabbedTranslationAdmin):
    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor')
        }
    }

    roles_model = models.UserRole

    entity_field = None

    list_display = ('name_translation', 'geo_id')


class CompanyOfficeInline(mixins.AdminPermissionMixin,
                          TranslationTabularInline):
    model = models.CompanyOffice

    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor'),
            'agencies': ('content_manager',)
        }
    }

    roles_model = models.UserRole

    entity_field = None

    exclude = (
        'proposed_by',
    )


class CompanyRepresentativeInline(mixins.AdminPermissionMixin,
                                  admin.TabularInline):
    model = models.CompanyRepresentative

    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor'),
            'agencies': ('content_manager',)
        }
    }

    roles_model = models.UserRole

    entity_field = None


@translate_list_display(models.Company)
@admin.register(models.Company)
class CompanyAdmin(mixins.AdminPermissionMixin,
                   ModerationAdmin,
                   TabbedTranslationAdmin):
    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor'),
            'agencies': ('content_manager',)
        },
        'add': {
            'common': ('superuser', 'asessor')
        }
    }

    roles_model = models.UserRole

    entity_field = 'agency'

    list_display = (
        'name_translation',
        'direct_budget',
        'site',
        'slug',
        'region',
        'is_partner',
    )
    inlines = [
        CompanyRepresentativeInline,
        CompanyOfficeInline
    ]


@admin.register(models.CompanyCertificate)
class CompanyCertificateAdmin(mixins.AdminPermissionMixin,
                              NameOnlyAdmin):
    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor')
        }
    }

    roles_model = models.UserRole

    entity_field = None


@admin.register(models.CompanyType)
class CompanyTypeAdmin(NameOnlyAdmin):
    pass


@translate_list_display(models.CompanyOffice)
@admin.register(models.CompanyOffice)
class CompanyOfficeAdmin(mixins.AdminPermissionMixin,
                         ModerationAdmin,
                         TabbedTranslationAdmin):
    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor')
        }
    }

    roles_model = models.UserRole

    entity_field = None

    list_display = (
        'company_name',
        'city',
        'address_translation',
        'is_main'
    )

    def company_name(self, obj):
        return get_translation(obj.company, 'name')
    company_name.short_description = 'Компания'

    fieldsets = (
        (None, {
            'fields': ('company',)
        }),
        ('Контакты', {
            'fields': ('email', 'phone')
        }),
        ('Расположение', {
            'fields': ('city', 'address', 'map', 'map_zoom', 'latitude', 'longitude')
        })
    )


@admin.register(models.Tld)
class TldAdmin(admin.ModelAdmin):
    list_display = ('value',)


admin.site.unregister(ModeratedObject)


@admin.register(ModeratedObject)
class PatchedModeratedObjectAdmin(mixins.AdminPermissionMixin,
                                  ModeratedObjectAdmin):
    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor')
        }
    }

    roles_model = models.UserRole

    entity_field = None


@admin.register(models.ExternalUser)
class ExternalUserAdmin(mixins.AdminPermissionMixin,
                        admin.ModelAdmin):
    permissions_dict = {
        'default': {
            'common': ('superuser', 'asessor'),
            'agencies': ('content_manager',)
        }
    }

    roles_model = models.UserRole

    entity_field = None

    list_display = ('username',)
