from typing import Optional, Callable

from django.contrib import admin

from commerce.adv_backend.backend.types import TModel
from commerce.adv_backend.backend.utils import get_translation


def create_translation_getter(
        name: str,
        field: str,
        short_description: Optional[str] = None
) -> Callable:
    def inner(self, model_instance: TModel) -> str:
        return get_translation(model_instance, field)

    inner.__name__ = name

    if short_description:
        inner.short_description = short_description

    return inner


def translate_list_display(model) -> Callable:
    def _model_admin_wrapper(admin_class: admin.ModelAdmin) -> admin.ModelAdmin:
        list_display = admin_class.list_display
        translation_fields = model.translation_fields

        for field in list_display:
            original_field = field.replace('_translation', '')

            if field.endswith('translation') and original_field in translation_fields:
                short_description = model._meta.get_field(original_field).verbose_name

                translation_getter = create_translation_getter(field, original_field, short_description)
                setattr(admin_class, translation_getter.__name__, translation_getter)
        return admin_class
    return _model_admin_wrapper
