from typing import Optional
from collections import defaultdict

from django.conf import settings

from .bunker import client as bunker
from .session import SessionWithBaseUrl


GEOBASE_LANGUAGES = ('en', 'ru', 'tr')


class Geobase:
    def __init__(self, base_url: str):
        self.session = SessionWithBaseUrl(base_url)
        self._translations = None

    @property
    def translations(self):
        if not self._translations:
            self._translations = defaultdict(dict)

            for section in ('by', 'cn', 'en', 'kz', 'ru', 'tr'):
                for node in bunker.fetch(f'sources/{section}/contact/agencies'):
                    language = {'by': 'ru', 'cn': 'zh_hans'}.get(section, section)

                    for country in node.get('citiesByCountries', []):
                        self._translations[country['geo_id']][language] = country['name']

                        for city in country.get('cities', []):
                            self._translations[city['geo_id']][language] = city['name']

        return self._translations

    def get_region_by_id(self, geo_id: int):
        response = self.session.get('region_by_id', params={'id': geo_id})

        return response.json()

    def get_region_parent_by_id(self,  geo_id: int) -> Optional[dict]:
        response = self.session.get('parents', params={'id': geo_id})
        parent_ids = response.json()

        for parent_id in parent_ids:
            parent = self.get_region_by_id(parent_id)

            if parent['type'] == 5:
                return parent

    def get_prepositional_case_by_id(self, geo_id: int, lang: str):
        response = self.session.get('linguistics_for_region', params={'id': geo_id, 'lang': lang})
        linguistics = response.json()

        preposition = linguistics['preposition']
        prepositional_name = linguistics['prepositional_case'] or linguistics['nominative_case']

        return preposition, prepositional_name

    def translate_region_name(self, geo_id: int):
        region = self.get_region_by_id(geo_id)

        translations = {}

        for language in settings.DB_LANGUAGES:
            translations[f'name_{language}'] = self.translations[geo_id].get(language)

        translations['name_ru'] = region['name']
        translations['name_en'] = region['en_name']

        for language in GEOBASE_LANGUAGES:
            preposition, prepositional_case = self.get_prepositional_case_by_id(geo_id, language)

            translations[f'preposition_{language}'] = preposition
            translations[f'name_prepositional_{language}'] = prepositional_case

        return translations


client = Geobase(
    base_url=settings.GEOBASE_BASE_URL
)
