from typing import Callable, Generic, Type, Optional

from django.conf import settings

from commerce.adv_backend.backend.types import TModel

from .geobase import client as geobase


class Model(Generic[TModel]):
    Model: Type[TModel] = None

    def __init__(self, **kwargs):
        self.model: TModel = self.Model()

        for name, value in kwargs.items():
            setattr(self.model, name, value)

    def create(self) -> TModel:
        if self.model.id is None:
            self.model.save()

        if hasattr(self.model, 'moderated_object'):
            self.model.moderated_object.approve()

        return self.model


class GeoModel(Model):
    def set(self, geo_id: int):
        try:
            self.model = self.Model.objects.get(geo_id=geo_id)
        except self.Model.DoesNotExist:
            pass

        self.model.geo_id = geo_id

        for k, v in geobase.translate_region_name(geo_id).items():
            setattr(self.model, k, v)


def noop_setter(_):
    pass


def simple_setter(name: str) -> Callable:
    def setter(self, value):
        setattr(self.model, name, value)

    return setter


def translations_setter(name: str, cb: Optional[Callable] = None) -> Callable:
    def setter(self, value):
        setattr(self.model, name, value['ru'] if not cb else cb(value['ru']))

        native = value.get('native')

        for language in settings.DB_LANGUAGES:
            model_value = value.get(language) or native

            if cb:
                model_value = cb(model_value)

            setattr(self.model, f'{name}_{language}', model_value)

    return setter


def model_setter(name: str, model_class) -> Callable:
    def setter(self, value):
        model = model_class()

        setattr(self.model, name, migrate(value, model))

    return setter


def map_setter() -> Callable:
    def setter(self, value):
        self.model.map = value['provider']
        self.model.latitude = float(value['latitude'])
        self.model.longitude = float(value['longitude'])
        self.model.map_zoom = int(value['zoom'] or 15)

    return setter


def migrate(from_data, model: Model[TModel]) -> TModel:
    if hasattr(model, 'set'):
        model.set(from_data)
    else:
        for key, value in from_data.items():
            getattr(model, f'set_{key}', noop_setter)(value)

    return model.create()
