import re
from operator import itemgetter

from commerce.adv_backend.backend import models

from .geobase import client as geobase
from .html import html_to_text

from .migrate import (
    GeoModel, Model,
    map_setter, model_setter, simple_setter, translations_setter
)


class City(GeoModel):
    Model = models.City


class Country(GeoModel):
    Model = models.Country


class Company(Model):
    Model = models.Company

    certificate_names = {
        'direct': 'Директ',
        'market': 'Маркет',
        'metrika': 'Метрика',
        'dialogs': 'Диалоги',
        'toloka': 'Толока'
    }

    certificate_preposition = 'по'
    certificate_preposition_en = 'by'

    certificate_names_prepositional = {
        'direct': 'Директу',
        'market': 'Маркету',
        'metrika': 'Метрике',
        'dialogs': 'Диалогам',
        'toloka': 'Толоке'
    }

    tld_countries = {
        'ru': {225},
        'by': {149},
        'cn': {134, 10105},
        'com': {96, 102, 124, 126, 134, 149, 159, 171, 180, 181, 204, 205, 206, 207, 210, 225, 246, 983, 10105, 20574},
        'com.tr': {983},
        'kz': {159}
    }

    set_description = translations_setter('description', html_to_text)
    set_logo = simple_setter('logo')
    set_name = translations_setter('name')
    set_site = simple_setter('site')

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        self.certificates = []
        self.tld = []
        self.representative = None

    def set_addresses(self, value: list):
        for address in value:
            for tld_value, countries_ids in self.tld_countries.items():
                if address['country'] in countries_ids:
                    tld, _ = models.Tld.objects.get_or_create(value=tld_value)

                    self.tld.append(tld)

        value = value[0]

        self.set_region(value['city'])

    def set_region(self, geo_id: int):
        region = geobase.get_region_parent_by_id(geo_id)

        if not region:
            return

        geo_id, name, name_en = itemgetter('id', 'name', 'en_name')(region)

        obj, _ = models.Region.objects.get_or_create(
            geo_id=geo_id,
            name=name,
            name_ru=name,
            name_en=name_en,
            name_tr=name_en,
            name_zh_hans=name_en
        )
        self.model.region = obj

    def set_certificates(self, value: dict):
        for code, is_certificated in value.items():
            name = self.certificate_names.get(code)
            name_prepositional = self.certificate_names_prepositional.get(code)

            if is_certificated:
                certificate, _ = models.CompanyCertificate.objects.get_or_create(
                    code=code,
                    name=name,
                    name_ru=name,
                    name_en=code.title(),
                    name_tr=code.title(),
                    name_zh_hans=code.title(),
                    preposition=self.certificate_preposition,
                    preposition_ru=self.certificate_preposition,
                    preposition_en=self.certificate_preposition_en,
                    preposition_tr=self.certificate_preposition_en,
                    preposition_zh_hans=self.certificate_preposition_en,
                    name_prepositional=name_prepositional,
                    name_prepositional_ru=name_prepositional,
                    name_prepositional_en=code.title(),
                    name_prepositional_tr=code.title(),
                    name_prepositional_zh_hans=code.title(),
                )

                self.certificates.append(certificate)

    def set_filters(self, value: list):
        if not value:
            return

        values = value[0]['values']
        values = [re.findall(r'\d', value) for value in values]
        values = [''.join(value) for value in values]

        self.model.direct_budget = min(values)

    def set_login_glv(self, value: str):
        self.representative, _ = models.ExternalUser.objects.get_or_create(username=value)

    def create(self):
        self.model.save()

        self.model.certificates.set(self.certificates)
        self.model.tld.set(self.tld)

        models.CompanyRepresentative.objects.get_or_create(
            representative=self.representative,
            company=self.model,
            is_main=True
        )

        return super().create()


class CompanyOffice(Model):
    Model = models.CompanyOffice

    set_address = translations_setter('address')
    set_city = model_setter('city', City)
    set_country = model_setter('country', Country)

    set_email = simple_setter('email')
    set_phone = simple_setter('phone')

    set_map = map_setter()
