from django.contrib.auth.models import User
from django.db.models import Model, Q


class AdminPermissionMixin:
    @property
    def entity_field(self) -> str:
        """
        Поле из таблички с ролями, на которое стоит смотреть
        для проверки пообъектных доступов, может быть опущено,
        если не хотим проверять доступ до объектов.
        """
        raise NotImplementedError('Entity field on roles should be implemented!')

    @property
    def permissions_dict(self) -> dict:
        """
        Словарь { action: { группа: список ролей } }
        В качестве алиаса для требования только группы (разрешены все роли)
        использовать пустой список или tuple.
        Если для всех действий одинаковые списки ролей, достаточно указать
        { default: { группа: список ролей } }
        Доступные действия:
        * view - просмотр объекта (или списка объектов этого типа)
        * change - изменение объекта (или списка объектов этого типа)
        * add - добавление объекта
        * module - отображение модели в списке на главной странице и доступ к списку объектов модели
        """
        raise NotImplementedError('Permissions dict field should be implemented!')

    @property
    def roles_model(self) -> Model:
        """
        Моделька для отслеживания ролей.
        """
        raise NotImplementedError('Roles model field should be implemented!')

    def get_queryset(self, request):
        entities = super().get_queryset(request)

        if self._has_superuser_or_asessor_role(request):
            return entities

        user = self._get_user(request)
        permission_groups = self._get_permission_groups('view')

        filter_condition = Q()
        for group in permission_groups:
            filter_condition |= Q(
                user=user,
                group=group,
                role__in=permission_groups[group]
            )

        roles = self._get_roles(filter_condition)

        if self.entity_field:
            entity_ids = roles.values_list(self.entity_field, flat=True)
            entities = entities.filter(pk__in=entity_ids)

        return entities

    def has_add_permission(self, request):
        return self._has_permission(request, 'add')

    def has_view_permission(self, request, obj=None):
        return self._has_permission(request, 'view', obj)

    def has_change_permission(self, request, obj=None):
        return self._has_permission(request, 'change', obj)

    def has_delete_permission(self, request, obj=None):
        return self._has_superuser_role(request)

    def has_module_permission(self, request):
        return self._has_permission(request, 'module')

    def _get_permission_groups(self, action):
        if action in self.permissions_dict:
            return self.permissions_dict[action]

        return self.permissions_dict['default']

    def _get_user(self, request):
        internal_login = request.yauser.login

        return User.objects.get(username=internal_login)

    def _get_roles(self, condition):
        return (
            self.roles_model.objects
            .filter(condition)
        )

    def _has_permission(self, request, action, obj=None):
        if self._has_superuser_or_asessor_role(request):
            return True

        permission_groups = self._get_permission_groups(action)

        if obj is None:
            filter_condition = Q(
                group__in=permission_groups.keys(),
                user=self._get_user(request)
            )

            return self._has_role(filter_condition)

        return self._has_obj_permission(request, action, obj)

    def _has_role(self, condition):
        return (
            self
            ._get_roles(condition)
            .exists()
        )

    def _has_superuser_or_asessor_role(self, request):
        filter_condition = Q(
            group='common',
            role__in=('superuser', 'asessor'),
            user=self._get_user(request)
        )

        return self._has_role(filter_condition)

    def _has_superuser_role(self, request):
        filter_condition = Q(
            group='common',
            role='superuser',
            user=self._get_user(request)
        )

        return self._has_role(filter_condition)

    def _has_obj_permission(self, request, action, obj):
        filter_condition = Q(user=self._get_user(request))
        permission_groups = self._get_permission_groups(action)

        for group in permission_groups:
            filter_condition |= Q(
                group=group,
                role__in=permission_groups[group]
            )

        if self.entity_field:
            filter_condition &= Q(**{self.entity_field: obj})

        return self._has_role(filter_condition)
