from django.db.models import Q
from rest_framework import serializers

from commerce.adv_backend.backend import models


def filter_param_list(param_list):
    return list(filter(bool, param_list))


class GeoFilterParams(serializers.Serializer):
    city = serializers.ListField(child=serializers.IntegerField(), required=False)
    country = serializers.ListField(child=serializers.IntegerField(), required=False)

    def preprocess(self):
        self.initial_data['city'] = filter_param_list(self.initial_data['city'])
        self.initial_data['country'] = filter_param_list(self.initial_data['country'])


class CertificatesFilterParams(serializers.Serializer):
    certificate = serializers.ListField(child=serializers.CharField(), required=False)

    def preprocess(self):
        self.initial_data['certificate'] = filter_param_list(self.initial_data['certificate'])


class CompaniesGroupFilterMixin:
    def filter_queryset(self, base_queryset):
        companies_ids = self._filter(base_queryset).values_list('id', flat=True)

        return base_queryset.filter(id__in=companies_ids)

    def _filter(self, base_queryset):
        queryset = super().filter_queryset(base_queryset)

        geo_filter_condition = self._get_geo_filter_condition()
        group = self.request.query_params.get('group', '')

        if group == 'main' or group == 'offices':
            return self._filter_with_certificates(queryset, geo_filter_condition, group)
        elif group == 'partners':
            return self._filter_partners(queryset, geo_filter_condition)
        else:
            return queryset.filter(geo_filter_condition)

    def _get_geo_filter_condition(self):
        city = self.request.query_params.getlist('city', [])
        country = self.request.query_params.getlist('country', [])

        params_serializer = GeoFilterParams(data={'city': city, 'country': country})

        params_serializer.preprocess()
        params_serializer.is_valid(raise_exception=True)

        city_ids = params_serializer.validated_data.get('city', [])
        country_ids = params_serializer.validated_data.get('country', [])

        geo_filter_kwargs = {}

        if city_ids:
            geo_filter_kwargs.update({'offices__city__geo_id__in': city_ids})

        if country_ids:
            geo_filter_kwargs.update({'offices__city__country__geo_id__in': country_ids})

        return Q(**geo_filter_kwargs)

    def _filter_with_certificates(self, queryset, geo_filter_condition, group):
        certificate = self.request.query_params.getlist('certificate', [])

        params_serializer = CertificatesFilterParams(data={'certificate': certificate})

        params_serializer.preprocess()
        params_serializer.is_valid(raise_exception=True)

        certificate_codes = params_serializer.validated_data.get('certificate', [])

        if certificate_codes:
            for code in certificate_codes:
                queryset = queryset.filter(certificates__code=code)

        main_office_condition = Q(offices__is_main=True)

        slugs_with_main_office = (
            queryset
            .filter(main_office_condition & geo_filter_condition)
            .values_list('slug', flat=True)
        )

        if group == 'main':
            return self._filter_main_offices(queryset, slugs_with_main_office, geo_filter_condition)
        else:
            return self._filter_other_offices(queryset, slugs_with_main_office, geo_filter_condition)

    def _filter_main_offices(self, queryset, slugs, geo_filter_condition):
        if self._has_regional_certificates():
            return queryset.filter(Q(slug__in=slugs) & geo_filter_condition)

        return queryset.filter(geo_filter_condition)

    def _filter_other_offices(self, queryset, slugs, geo_filter_condition):
        if not self._has_regional_certificates():
            return queryset.none()

        return queryset.filter(~Q(slug__in=slugs) & geo_filter_condition)

    def _has_regional_certificates(self):
        certificate_codes = self.request.query_params.getlist('certificate', [])
        regional_certificates = (
            models.CompanyCertificate.objects
            .filter(code__in=certificate_codes, is_regional=True)
        )

        return regional_certificates.exists()

    def _filter_partners(self, queryset, geo_filter_condition):
        return queryset.filter(Q(is_partner=True) & geo_filter_condition)


class CompaniesRepresentativeFilterMixin:
    def filter_queryset(self, queryset):
        queryset = super().filter_queryset(queryset)

        external_login = self.request.headers.get('x-ya-external-login')

        return queryset.filter(representatives__username=external_login)
