from html import unescape

from django.conf import settings
from django.db import models
from django.utils.html import strip_tags
from django.contrib.postgres import fields

from slugify import slugify

from .abstract import ModeratedModel
from ..mixins import TranslationRequiredMixin
from ..validators import MaterialsUrlValidator


class Company(ModeratedModel, TranslationRequiredMixin):
    region = models.ForeignKey(
        verbose_name='Регион',
        to='Region',
        on_delete=models.PROTECT,
        blank=True,
        null=True
    )
    name = models.CharField(
        verbose_name='Название',
        max_length=128,
        blank=True
    )
    slug = models.SlugField(
        verbose_name='Слаг',
        max_length=128,
        blank=True,
        unique=True
    )
    logo = models.URLField(
        verbose_name='Ссылка на логотип',
        blank=True
    )
    types = models.ManyToManyField(
        verbose_name='Типы',
        to='CompanyType',
        blank=True
    )
    certificates = models.ManyToManyField(
        verbose_name='Сертификаты',
        to='CompanyCertificate',
        blank=True
    )
    direct_company_type = models.CharField(
        verbose_name='Тип компании с сертификатом по Директу',
        choices=settings.DIRECT_COMPANY_TYPES,
        max_length=20,
        blank=True,
        null=True
    )
    direct_budget = models.PositiveIntegerField(
        verbose_name='Минимальный бюджет рекламной кампании в Директе',
        blank=True,
        null=True
    )
    site = models.CharField(
        verbose_name='Сайт',
        unique=True,
        max_length=200
    )
    representatives = models.ManyToManyField(
        to='ExternalUser',
        through='CompanyRepresentative',
        verbose_name='Представители'
    )
    description = models.TextField(
        verbose_name='Описание компании',
        blank=True
    )
    tld = models.ManyToManyField(
        verbose_name='Где показывать',
        to='Tld'
    )
    materials = fields.ArrayField(
        verbose_name='Ссылки на связанные материалы',
        base_field=models.URLField(validators=[MaterialsUrlValidator()]),
        blank=True,
        default=list
    )
    is_partner = models.BooleanField(
        verbose_name='Является ли партнером',
        default=False
    )
    is_active = models.BooleanField(
        verbose_name='Показывать ли в списке агентств',
        default=False
    )

    translation_fields = (
        'description',
        'materials',
        'name'
    )
    required_translation_fields = (
        'name',
    )

    def make_slug(self):
        name = None
        for lang in settings.DB_LANGUAGES:
            name = getattr(self, f'name_{lang}')
            if name:
                break

        sanitized_name = strip_tags(unescape(name))
        slug = slugify(sanitized_name, max_length=100)

        if Company.objects.filter(slug=slug).values('slug'):
            tld = self.site.split(".")[-1]
            self.slug = f'{slug}-{tld}'
        else:
            self.slug = slug

    def save(self, *args, **kwargs):
        if not self.slug:
            self.make_slug()

        super().save(*args, **kwargs)

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = 'Компания'
        verbose_name_plural = 'Компании'

        indexes = [
            models.Index(fields=['slug']),
            models.Index(fields=['direct_budget'])
        ]
