from django.db import models

from .abstract import ModeratedModel
from ..mixins import TranslationRequiredMixin


MAP_CHOICES = [
    ('yandex', 'Яндекс.Карты'),
    ('osm', 'OpenStreetMap')
]


class CompanyOffice(ModeratedModel, TranslationRequiredMixin):
    is_main = models.BooleanField(
        verbose_name='Головной офис',
        blank=True,
        default=False
    )
    city = models.ForeignKey(
        verbose_name='Город',
        to='City',
        on_delete=models.PROTECT
    )
    address = models.CharField(
        verbose_name='Адрес',
        max_length=256,
        blank=True
    )
    phone = models.CharField(
        verbose_name='Номер телефона',
        max_length=128,
        blank=True
    )
    email = models.EmailField(
        verbose_name='Электронная почта',
        blank=True,
        null=True
    )
    map = models.CharField(
        verbose_name='Тип карты',
        choices=MAP_CHOICES,
        max_length=6
    )
    latitude = models.DecimalField(
        verbose_name='Широта',
        max_digits=9,
        decimal_places=6,
        blank=True,
        null=True
    )
    longitude = models.DecimalField(
        verbose_name='Долгота',
        max_digits=9,
        decimal_places=6,
        blank=True,
        null=True
    )
    map_zoom = models.IntegerField(
        verbose_name='Масштаб карты'
    )
    company = models.ForeignKey(
        verbose_name='Компания',
        to='Company',
        related_name='offices',
        on_delete=models.PROTECT
    )

    translation_fields = ('address',)
    required_translation_fields = ()

    def save(self, *args, **kwargs):
        self.proposed_by = kwargs.pop('proposed_by', self.company.proposed_by)

        super().save(*args, **kwargs)

    class Meta:
        verbose_name = 'Офис компании'
        verbose_name_plural = 'Офисы компании'

        indexes = [
            models.Index(fields=['is_main'])
        ]

        ordering = [
            '-is_main',
            '-city__display_priority',
            'city__name'
        ]
