from django.conf import settings
from django.contrib.auth.models import User
from django.dispatch import receiver

from moderation import moderation, moderator
from moderation.signals import post_moderation

from . import libs, models, utils


class CompanyModerator(moderator.GenericModerator):
    auto_approve_for_superusers = True
    auto_approve_for_staff = False

    notify_moderator = True
    notify_user = False

    message_backend_class = libs.StartrekMessageBackend

    message_template_moderator = 'message_moderator.txt'
    subject_template_moderator = 'subject_moderator.txt'

    visibility_column = 'is_approved'

    fields_exclude = (
        'proposed_by',
        'is_approved'
    )

    def send(self, content_object, *args, **kwargs):
        name = ''
        if hasattr(content_object, 'name'):
            name = utils.get_translation(content_object, 'name')

        if content_object.proposed_by:
            super().send(content_object, *args, **kwargs, extra_context={
                'admin_base_url': settings.ADMIN_BASE_URL,
                'exists': type(content_object).objects.filter(pk=content_object.pk).exists(),
                'name': name,
                'proposed_by': content_object.proposed_by
            })

    def is_auto_approve(self, obj, user):
        roles = (
            models.UserRole.objects
            .filter(
                group='common',
                role__in=('superuser', 'asessor'),
                user=User.objects.get(username=user)
            )
        )

        if roles.exists():
            return f'Auto-approved by {user}'

        return False


@receiver(post_moderation)
def clear_proposed_by(sender, instance, status, **kwargs):
    if hasattr(instance, 'proposed_by'):
        instance.proposed_by = ''
        instance.save()


moderation.register(models.Company, CompanyModerator)
moderation.register(models.CompanyOffice, CompanyModerator)
