from django.conf import settings
from django.contrib.auth import get_user_model

from django_idm_api.hooks import BaseHooks
from django_idm_api.exceptions import RoleNotFound, UserNotFound

from ..models import Company, UserRole


def get_agency_subtree(agency):
    data = {
        'name': {
            'en': agency.name_en,
            'ru': agency.name_ru
        },
        'roles': {
            'slug': 'role',
            'name': {
                'en': 'Role',
                'ru': 'Роль'
            },
            'help': {
                'en': 'Role effective inside the Agencies group',
                'ru': 'Роль внутри группы "Агентства"'
            },
            'values': {}
        }
    }

    values = data['roles']['values']

    for role, role_info in settings.AGENCIES_ROLES:
        values[role] = dict(role_info, **{'set': role})
    return {agency.slug: data}


class Hooks(BaseHooks):
    def info(self, **kwargs):
        """Возвращает роли Админки Сайта Рекламы"""
        data = {
            'code': 0,
            'roles': {}
        }

        # вводим иерархичность дерева ролей
        data['roles']['slug'] = 'group'
        data['roles']['name'] = {
            'en': 'Type',
            'ru': 'Тип'
        }
        roles = data['roles']['values'] = {}

        # добавляем группу Общие роли
        roles['common'] = {
            'name': {
                'en': 'Common',
                'ru': 'Общие'
            },
            'help': {
                'en': 'Global roles',
                'ru': 'Глобальные роли'
            },
            'roles': {
                'slug': 'role',
                'name': {
                    'en': 'Role',
                    'ru': 'Роль'
                },
                'values': {}
            }
        }

        for role, global_role_info in settings.COMMON_ROLES:
            roles['common']['roles']['values'][role] = global_role_info

        # добавляем группу Агентсва
        companies = Company.objects.all()
        if companies.exists():
            roles['agencies'] = {
                'name': {
                    'en': 'Agencies',
                    'ru': 'Агентства'
                },
                'roles': {
                    'slug': 'agency_on',
                    'name': {
                        'en': 'Agency',
                        'ru': 'Агентство'
                    },
                    'values': {}
                }
            }

        for company in companies:
            agency_subtree = get_agency_subtree(company)
            roles['agencies']['roles']['values'].update(agency_subtree)
        return data

    def add_role_impl(self, login, role, fields, **kwargs):
        role_name = role['role']
        if role_name not in settings.ALL_ROLES:
            raise RoleNotFound('Role does not exist: %s' % role_name)

        group = role['group']
        user = self._get_or_create_user(login)
        agency = None

        if group == 'common':
            if role_name == 'superuser':
                user.is_superuser = True
            elif role_name == 'asessor':
                pass
            else:
                raise RoleNotFound('Role common does not exist: %s' % role_name)
        elif group == 'agencies':
            try:
                agency = Company.objects.get(slug=role['agency_on'])
            except Company.DoesNotExist:
                raise RoleNotFound('Role agency does not exist: %s' % role['agency_on'])
        else:
            raise RoleNotFound('Role group not exist: %s' % group)

        user.is_staff = True
        user.save(update_fields=('is_staff', 'is_superuser'))

        UserRole.objects.get_or_create(
            user=user,
            group=group,
            role=role_name,
            agency=agency
        )

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        role_name = role['role']
        if not role_name:
            raise RoleNotFound('Role does not exist: %s' % role_name)

        user = self._get_user(login)
        group = role['group']
        agency = None

        if group == 'common':
            if role_name == 'superuser':
                user.is_superuser = False
        elif group == 'agencies':
            try:
                agency = Company.objects.get(slug=role['agency_on'])
            except Company.DoesNotExist:
                return
        else:
            raise RoleNotFound('Role group not exist: %s' % group)

        user_has_roles = (
            UserRole.objects
            .filter(user=user)
            .exclude(
                group=group,
                role=role_name,
                agency=agency
            )
            .exists()
        )

        if not user_has_roles:
            user.is_staff = False
            user.save(update_fields=('is_staff', 'is_superuser'))

        (
            UserRole.objects
            .filter(
                user=user,
                group=group,
                role=role_name,
                agency=agency
            )
            .delete()
        )

    def get_roles(self, request):
        page = int(request.GET.get('page', 0))
        begin_index = page * settings.ROLES_PAGE_SIZE

        roles = (
            UserRole.objects
            .prefetch_related('user', 'agency')
            .order_by('id')
            [begin_index:begin_index + settings.ROLES_PAGE_SIZE]
        )

        data = {'code': 0, 'roles': []}

        for user_role in roles:
            path_pattern = settings.PATH_PATTERNS[user_role.group][user_role.role]

            path = path_pattern
            if user_role.group != 'common':
                path %= {'entity_id': user_role.agency.slug}

            data['roles'].append({
                'login': user_role.user.username,
                'path': path
            })

        next_page = page + 1
        if len(data['roles']) == settings.ROLES_PAGE_SIZE:
            data['next-url'] = '%s?page=%d' % (request.path, next_page)

        return data

    # Вспомогательные методы
    def _get_user(self, login):
        """Получить пользователя по логину или кинуть эксепшн."""
        try:
            return get_user_model().objects.get(username=login)
        except get_user_model().DoesNotExist:
            raise UserNotFound('User does not exist: %s' % login)

    def _get_or_create_user(self, login):
        """Получить существующего пользователя по логину или создать нового."""
        try:
            user = self._get_user(login)
        except UserNotFound:
            user = get_user_model().objects.create(username=login)
        return user
