from django.conf import settings
from django.core.validators import FileExtensionValidator

from rest_framework import serializers

from commerce.adv_backend.backend import models

from ..validators import FileValidator, MaterialsUrlValidator
from .company_certificate import CompanyCertificateSerializer
from .company_office import CitySerializer, CompanyOfficeSerializer, CompanyOfficeWriteSerializer
from .company_representative import CompanyRepresentativeSerializer
from .company_type import CompanyTypeSerializer
from .country import CountrySerializer


class CompanySerializer(serializers.ModelSerializer):
    certificates = CompanyCertificateSerializer(many=True)
    materials = serializers.ListField(child=serializers.URLField())
    offices = CompanyOfficeSerializer(many=True)
    representatives = CompanyRepresentativeSerializer(source='companyrepresentative_set', many=True)
    types = CompanyTypeSerializer(many=True)

    class Meta:
        model = models.Company
        fields = (
            'certificates',
            'description',
            'direct_budget',
            'direct_company_type',
            'logo',
            'materials',
            'name',
            'offices',
            'representatives',
            'site',
            'slug',
            'tld',
            'types',
            'is_partner',
            'is_active'
        )
        depth = 1


class CompanyWriteSerializer(serializers.ModelSerializer):
    name = serializers.CharField(
        required=True
    )
    proposed_by = serializers.CharField(
        required=True
    )
    types = serializers.SlugRelatedField(
        queryset=models.CompanyType.objects.all(),
        slug_field='code',
        many=True
    )
    certificates = serializers.SlugRelatedField(
        queryset=models.CompanyCertificate.objects.all(),
        slug_field='code',
        many=True
    )
    representatives = serializers.SlugRelatedField(
        queryset=models.ExternalUser.objects.all(),
        slug_field='username',
        many=True
    )
    tld = serializers.SlugRelatedField(
        queryset=models.Tld.objects.all(),
        slug_field='value',
        many=True
    )
    materials = serializers.ListField(
        child=serializers.URLField(validators=[MaterialsUrlValidator()]),
        required=False
    )
    offices = CompanyOfficeWriteSerializer(
        many=True,
        required=False
    )

    class Meta:
        model = models.Company
        fields = [
            'certificates',
            'description',
            'direct_budget',
            'logo',
            'materials',
            'name',
            'offices',
            'proposed_by',
            'representatives',
            'site',
            'slug',
            'tld',
            'types'
        ]
        depth = 1

    def create(self, validated_data):
        offices_data = validated_data.pop('offices', [])
        company = super().create(validated_data)

        for office_data in offices_data:
            models.CompanyOffice.objects.create(company=company, **office_data)

        return company

    def update(self, instance, validated_data):
        offices_data = validated_data.pop('offices', [])
        instance.offices.set([])

        for office_data in offices_data:
            models.CompanyOffice.objects.create(company=instance, **office_data)

        return super().update(instance, validated_data)


class CompanyFiltersSerializer(serializers.Serializer):
    certificates = CompanyCertificateSerializer(many=True)

    def __new__(cls, *args, **kwargs):
        kwargs.pop('many', None)  # Насильно сериализуем весь список целиком
        return super(CompanyFiltersSerializer, cls).__new__(cls, *args, **kwargs)

    def to_representation(self, companies):
        request = self.context.get('request')

        certificates_codes = (
            companies
            .values_list('certificates__code', flat=True)
            .distinct('certificates__code')
        )

        cities_indexes = (
            companies
            .values_list('offices__city__geo_id', flat=True)
            .distinct('offices__city__geo_id')
        )

        countries_indexes = (
            companies
            .values_list('offices__city__country__geo_id', flat=True)
            .distinct('offices__city__country__geo_id')
        )

        certificates = CompanyCertificateSerializer(
            data=models.CompanyCertificate.objects
            .filter(code__in=certificates_codes),
            many=True,
            context={'request': request}
        )

        cities = CitySerializer(
            data=models.City.objects
            .filter(geo_id__in=cities_indexes),
            many=True,
            context={'request': request}
        )

        countries = CountrySerializer(
            data=models.Country.objects
            .filter(geo_id__in=countries_indexes),
            many=True,
            context={'request': request}
        )

        # Нужно вызывать перед тем, как доставать data
        certificates.is_valid()
        cities.is_valid()
        countries.is_valid()

        return dict(
            certificates=certificates.data,
            cities=cities.data,
            countries=countries.data
        )


class CompaniesFormSerializer(serializers.Serializer):
    validator_file_mime = FileValidator(settings.ALLOWED_CONTENT_TYPES)
    validator_file_ext = FileExtensionValidator(allowed_extensions=settings.ALLOWED_EXTENSIONS, code='400_IEF')
    validators_file = [validator_file_ext, validator_file_mime]

    service = serializers.PrimaryKeyRelatedField(
        label='Сервис',
        required=True,
        queryset=models.CompanyCertificate.objects.all()
    )
    add_list = serializers.FileField(label='Кому добавить?', required=True, validators=validators_file)
    remove_list = serializers.FileField(label='У кого забрать?', required=True, validators=validators_file)


class ExternalUserCompaniesListSerializer(serializers.Serializer):
    def to_representation(self, instance):
        return {
            'slug': instance.slug,
            'name': instance.name,
            'is_active': instance.is_active
        }


class CompaniesDownloadPageSerializer(serializers.Serializer):
    tld = serializers.SlugRelatedField(
        label='Домен',
        slug_field='value',
        required=True,
        queryset=models.Tld.objects.all()
    )


class CompaniesDownloadCSVSerializer(serializers.ModelSerializer):
    certificates = CompanyCertificateSerializer(many=True)
    offices = CompanyOfficeSerializer(many=True)
    representatives = CompanyRepresentativeSerializer(source='companyrepresentative_set', many=True)

    class Meta:
        model = models.Company
        fields = (
            'id',
            'certificates',
            'name',
            'is_active',
            'is_partner',
            'offices',
            'region',
            'representatives',
            'slug'
        )
        depth = 1
