import re
from urllib.parse import urlparse

from django.core.exceptions import ValidationError
from django.utils.deconstruct import deconstructible


AVAILABLE_HOSTS = [
    'yandex-ad.cn',
    'yandex.by',
    'yandex.com',
    'yandex.com.tr',
    'yandex.kz',
    'yandex.ru'
]
PATH_RE = re.compile('^/adv/(news|solutions/cases)/([\w-]+)/?$')


@deconstructible
class MaterialsUrlValidator:
    message = "Некорректная ссылка на материал: %(link)s."
    code = '400_IMU'

    def __call__(self, value):
        url = urlparse(value)
        hostname = url.hostname or ''
        path = PATH_RE.match(url.path or '')

        is_correct_scheme = url.scheme == 'https'
        is_correct_host = hostname in AVAILABLE_HOSTS
        is_correct_path = bool(path)

        is_correct_url = is_correct_scheme and is_correct_host and is_correct_path
        if not is_correct_url:
            params = {'link': value}
            raise ValidationError(self.message, code=self.code, params=params)

    def __eq__(self, other):
        return (
            isinstance(other, MaterialsUrlValidator) and
            other.message == self.message and
            other.code == self.code
        )
