from django.conf import settings
from django.db.models import Q
from django_filters import rest_framework as filters
from django.http import HttpResponse
from django.utils import translation

import csv
import codecs

from rest_framework.mixins import ListModelMixin, CreateModelMixin, RetrieveModelMixin, UpdateModelMixin
from rest_framework.parsers import FormParser, FileUploadParser, MultiPartParser
from rest_framework.renderers import TemplateHTMLRenderer
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework.viewsets import GenericViewSet

from commerce.adv_backend.backend import models, serializers, utils, permissions
from commerce.adv_backend.backend.decorators import require_query_params
from commerce.adv_backend.backend.mixins import (
    BudgetOrderMixin,
    CompaniesGroupFilterMixin,
    CompaniesRepresentativeFilterMixin,
    ViewSuperuserPermissionMixin
)
from commerce.adv_backend.backend.utils import get_admin_link


class CompanyFilter(utils.TldFilter):
    budget = filters.NumberFilter(
        field_name='direct_budget',
        lookup_expr='lte'
    )
    slug = filters.AllValuesMultipleFilter(
        field_name='slug'
    )
    site = filters.CharFilter(
        field_name='site'
    )


class BaseCompanyViewSet(ListModelMixin,
                         GenericViewSet):
    filterset_class = CompanyFilter

    @require_query_params('tld')
    def get_queryset(self):
        return (
            models.Company.objects
            .filter(is_active=True)
            .prefetch_related(
                'certificates',
                'offices',
                'offices__city',
                'offices__city__country',
                'tld'
            )
            .all()
        )

    @require_query_params('tld')
    def get_serializer_class(self):
        if self.request.method == 'GET':
            return serializers.CompanySerializer

        return serializers.CompanyWriteSerializer


class BaseCompanyWithOrderingViewSet(BaseCompanyViewSet):
    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        ordered_queryset = self.order_queryset(queryset)

        page = self.paginate_queryset(ordered_queryset)
        if page is not None:
            serializer = self.get_serializer(page, many=True)
            return self.get_paginated_response(serializer.data)

        serializer = self.get_serializer(ordered_queryset, many=True)
        return Response(serializer.data)


class CompanyViewSet(CompaniesGroupFilterMixin,
                     BudgetOrderMixin,
                     BaseCompanyWithOrderingViewSet,
                     CreateModelMixin,
                     RetrieveModelMixin,
                     UpdateModelMixin):
    permission_classes = [permissions.IsRepresentative]
    lookup_field = 'slug'


class CompaniesSlugsList(CompaniesGroupFilterMixin,
                         BudgetOrderMixin,
                         BaseCompanyWithOrderingViewSet):
    def get_serializer_class(self):
        return utils.SlugSerializer


class CompaniesFilters(BaseCompanyViewSet):
    @require_query_params('tld')
    def get_queryset(self):
        return (
            models.Company.objects
            .filter(
                Q(is_partner=True) | Q(certificates__isnull=False),
                is_active=True
            )
            .prefetch_related(
                'certificates',
                'offices',
                'offices__city',
                'offices__city__country',
                'tld'
            )
            .all()
        )

    def get_serializer_class(self):
        return serializers.CompanyFiltersSerializer


class CompaniesFormView(APIView):
    renderer_classes = [TemplateHTMLRenderer]
    parser_classes = [FormParser, MultiPartParser, FileUploadParser]
    permission_classes = [ViewSuperuserPermissionMixin]

    def post(self, request):
        serializer = serializers.CompaniesFormSerializer(data=request.data)

        if not serializer.is_valid():
            return Response({
                'serializer': serializer,
                'button_text': 'Отправить',
                'title': 'Форма загрузки компаний',
                'url_name': 'v1:companies-certificates'
            }, template_name='company/form.html')

        return Response({
            'message': 'Форма успешно отправлена',
            'title': 'Форма загрузки компаний'
        }, template_name='company/form_success.html')

    def get(self, _request):
        serializer = serializers.CompaniesFormSerializer()

        return Response({
            'serializer': serializer,
            'button_text': 'Отправить',
            'title': 'Форма загрузки компаний',
            'url_name': 'v1:companies-certificates'
        }, template_name='company/form.html')


class ExternalUserCompaniesListFilters(utils.TldFilter):
    pass


class ExternalUserCompaniesList(CompaniesRepresentativeFilterMixin,
                                ListModelMixin,
                                GenericViewSet):
    filterset_class = ExternalUserCompaniesListFilters

    @require_query_params('tld')
    def get_queryset(self):
        return (
            models.Company.objects
            .prefetch_related('tld')
            .all()
        )

    def get_serializer_class(self):
        return serializers.ExternalUserCompaniesListSerializer


class CompaniesDownloadView(APIView):
    renderer_classes = [TemplateHTMLRenderer]
    parser_classes = [FormParser]
    permission_classes = [ViewSuperuserPermissionMixin]

    def get(self, _request):
        serializer = serializers.CompaniesDownloadPageSerializer()

        return Response({
            'serializer': serializer,
            'button_text': 'Выгрузить',
            'title': 'Выгрузка компаний',
            'url_name': 'v1:companies-download'
        }, template_name='company/download.html')

    def _has_certificate(self, row, certificate_code):
        for certificate in row['certificates']:
            if dict(certificate)['code'] == certificate_code:
                return 'да'
        else:
            return 'нет'

    def _get_representatives(self, row):
        if 'representatives' not in row:
            return ''

        usernames = []

        for user in row['representatives']:
            username = user['username']

            if username:
                usernames.append(username)

        return ', '.join(usernames)

    def _get_cities(self, row):
        cities = []

        for office in row['offices']:
            office_json = dict(office)

            if office_json['is_main']:
                city = office_json['city']
                city_name = dict(city)['name']

                # Если нет перевода на нужный язык
                if city_name and city_name not in cities:
                    cities.append(city_name)

        return ', '.join(cities)

    def _get_region(self, row):
        if 'region' in row and row['region']:
            return dict(row['region'])['name']
        else:
            return ''

    def _get_bool_field(self, row, field_name):
        if row[field_name]:
            return 'да'
        else:
            return 'нет'

    def _get_company_link(self, row):
        id = row['id']

        return get_admin_link(id, 'company')

    def build_csv_data(self, tld):
        csv_header = {
            'representatives': 'Главные представители',
            'title': 'Название',
            'slug': 'Слаг',
            'cities': 'Города',
            'region': 'Регион',
            'direct': 'Директ',
            'metrika': 'Метрика',
            'market': 'Маркет',
            'dialogs': 'Диалоги',
            'toloka': 'Толока',
            'is_active': 'Активно ли агентство',
            'is_partner': 'Является ли партнером',
            'company_admin_link': 'Ссылка на компанию в админке'
        }
        csv_data = [csv_header]

        queryset = (
            models.Company.objects
            .filter(tld__value=tld)
            .prefetch_related(
                'certificates',
                'representatives',
                'offices',
                'offices__city'
            )
            .order_by('slug')
            .distinct('slug')
        )

        serializer = serializers.CompaniesDownloadCSVSerializer(queryset, many=True)
        data = serializer.data

        companies_data = []

        for company_row in data:
            companies_data.append(dict(company_row))

        for row in companies_data:
            csv_data.append({
                'representatives': self._get_representatives(row),
                'title': row['name'],
                'slug': row['slug'],
                'cities': self._get_cities(row),
                'region': self._get_region(row),
                'direct': self._has_certificate(row, 'direct'),
                'metrika': self._has_certificate(row, 'metrika'),
                'market': self._has_certificate(row, 'market'),
                'dialogs': self._has_certificate(row, 'dialogs'),
                'toloka': self._has_certificate(row, 'toloka'),
                'is_active': self._get_bool_field(row, 'is_active'),
                'is_partner': self._get_bool_field(row, 'is_partner'),
                'company_admin_link': self._get_company_link(row)
            })

        return csv_data

    def post(self, request):
        serializer = serializers.CompaniesDownloadPageSerializer(data=request.data)

        if not serializer.is_valid():
            return Response({
                'fields': 'tld',
            }, template_name='400.html')

        tld = serializer.data['tld']

        language = settings.TLD_TO_DB_LANG.get(tld)

        translation.activate(language)
        request.LANGUAGE_CODE = translation.get_language()

        csv_rows = self.build_csv_data(tld)

        response = HttpResponse(content_type='text/csv')
        response['Content-Disposition'] = f'attachment; filename=companies_{tld}.csv'

        # Force response to be UTF-8
        response.write(codecs.BOM_UTF8)

        header = [
            'representatives',
            'title',
            'slug',
            'cities',
            'region',
            'direct',
            'metrika',
            'market',
            'dialogs',
            'toloka',
            'is_active',
            'is_partner',
            'company_admin_link'
        ]

        writer = csv.DictWriter(response, delimiter=';', fieldnames=header)

        for row in csv_rows:
            writer.writerow(row)

        return response
