package cache

import (
	"crypto/md5"
	"encoding/hex"
	"strings"

	"a.yandex-team.ru/commerce/blogs_pumpkin/config"
	"a.yandex-team.ru/commerce/libs/goblogs"
)

const (
	blogInfoKey       = "info"
	blogCategoriesKey = "categories"
	blogPostsKey      = "posts"
	blogTagsKey       = "tags"

	postInfoKey            = "info"
	postCommentsKey        = "comments"
	postRelatedArticlesKey = "related-articles"

	blogsStatusKey = "blogs-status"

	BlogsOK BlogsStatus = iota
	BlogsNotOK
)

var (
	DefaultCache Cache
)

func init() {
	switch config.Values.Cache.Type {
	case "local":
		DefaultCache = NewLocalCache()

	case "redis":
		DefaultCache = NewRedisCache()

	default:
		panic("unknown cache type, check your config")
	}
}

type BlogsStatus int

type Cache interface {
	GetBlog(slug, language string) (*goblogs.Blog, bool)
	SetBlog(slug, language string, blog *goblogs.Blog) error

	GetBlogCategories(slug, language string) ([]goblogs.Category, bool)
	SetBlogCategories(slug, language string, categories []goblogs.Category) error

	GetBlogPosts(slug, language string) ([]goblogs.Post, bool)
	SetBlogPosts(slug, language string, posts []goblogs.Post) error

	GetBlogTags(slug, language string) ([]goblogs.Tag, bool)
	SetBlogTags(slug, language string, tags []goblogs.Tag) error

	GetPost(blogSlug, language, slug string) (*goblogs.Post, bool)
	SetPost(blogSlug, language, slug string, post *goblogs.Post) error

	GetPostComments(blogSlug, language, slug string) ([]goblogs.Comment, bool)
	SetPostComments(blogSlug, language, slug string, comments []goblogs.Comment) error

	GetPostRelatedArticles(blogSlug, language, slug string) ([]goblogs.RelatedArticle, bool)
	SetPostRelatedArticles(blogSlug, language, slug string, relatedArticles []goblogs.RelatedArticle) error

	GetBlogsStatus() BlogsStatus
	SetBlogsStatus(value BlogsStatus) error

	Lock(key string) error
	Unlock(key string) error
}

func GetCacheKey(params ...string) string {
	params = append(params, config.Values.Cache.Version)

	s := strings.Join(params, ":")
	b := md5.Sum([]byte(s))

	return hex.EncodeToString(b[:])
}
