package cache

import (
	"context"
	"encoding/json"
	"time"

	"github.com/go-redis/redis/v8"

	"a.yandex-team.ru/commerce/blogs_pumpkin/config"
	"a.yandex-team.ru/commerce/blogs_pumpkin/libs/redislock"
	"a.yandex-team.ru/commerce/libs/goblogs"
)

var (
	redisClient = redis.NewUniversalClient(&redis.UniversalOptions{
		Addrs:      config.Values.Redis.Addresses,
		Password:   config.Values.Redis.Password,
		MasterName: config.Values.Redis.MasterName,
		DB:         0,
		ReadOnly:   false,
	})
)

type redisCache struct {
	redis.UniversalClient

	locks map[string]*redislock.Lock
}

func NewRedisCache() Cache {
	return &redisCache{
		UniversalClient: redisClient,

		locks: map[string]*redislock.Lock{},
	}
}

func (c *redisCache) get(result interface{}, params ...string) bool {
	key := GetCacheKey(params...)

	value, err := c.Get(context.TODO(), key).Result()

	if err != nil {
		return false
	}

	return json.Unmarshal([]byte(value), result) == nil
}

func (c *redisCache) set(value interface{}, expiration time.Duration, params ...string) error {
	key := GetCacheKey(params...)

	s, err := json.Marshal(value)

	if err != nil {
		return err
	}

	if expiration == 0 {
		expiration = time.Duration(config.Values.Redis.TTL)
	}

	return c.Set(context.TODO(), key, s, expiration).Err()
}

func (c *redisCache) GetBlog(slug, language string) (*goblogs.Blog, bool) {
	var blog goblogs.Blog

	inCache := c.get(&blog, slug, language, blogInfoKey)

	return &blog, inCache
}

func (c *redisCache) SetBlog(slug, language string, blog *goblogs.Blog) error {
	return c.set(blog, 0, slug, language, blogInfoKey)
}

func (c *redisCache) GetBlogCategories(slug, language string) ([]goblogs.Category, bool) {
	var categories []goblogs.Category

	inCache := c.get(&categories, slug, language, blogCategoriesKey)

	return categories, inCache
}

func (c *redisCache) SetBlogCategories(slug, language string, categories []goblogs.Category) error {
	return c.set(categories, 0, slug, language, blogCategoriesKey)
}

func (c *redisCache) GetBlogPosts(slug, language string) ([]goblogs.Post, bool) {
	var posts []goblogs.Post

	inCache := c.get(&posts, slug, language, blogPostsKey)

	return posts, inCache
}

func (c *redisCache) SetBlogPosts(slug, language string, posts []goblogs.Post) error {
	return c.set(posts, 0, slug, language, blogPostsKey)
}

func (c *redisCache) GetBlogTags(slug, language string) ([]goblogs.Tag, bool) {
	var tags []goblogs.Tag

	inCache := c.get(&tags, slug, language, blogTagsKey)

	return tags, inCache
}

func (c *redisCache) SetBlogTags(slug, language string, tags []goblogs.Tag) error {
	return c.set(tags, 0, slug, language, blogTagsKey)
}

func (c *redisCache) GetPost(blogSlug, language, slug string) (*goblogs.Post, bool) {
	var post goblogs.Post

	inCache := c.get(&post, blogSlug, language, slug, postInfoKey)

	return &post, inCache
}

func (c *redisCache) SetPost(blogSlug, language, slug string, post *goblogs.Post) error {
	return c.set(post, 0, blogSlug, language, slug, postInfoKey)
}

func (c *redisCache) GetPostComments(blogSlug, language, slug string) ([]goblogs.Comment, bool) {
	var comments []goblogs.Comment

	inCache := c.get(&comments, blogSlug, language, slug, postCommentsKey)

	return comments, inCache
}

func (c *redisCache) SetPostComments(blogSlug, language, slug string, comments []goblogs.Comment) error {
	return c.set(comments, 0, blogSlug, language, slug, postCommentsKey)
}

func (c *redisCache) GetPostRelatedArticles(blogSlug, language, slug string) ([]goblogs.RelatedArticle, bool) {
	var relatedArticles []goblogs.RelatedArticle

	inCache := c.get(&relatedArticles, blogSlug, language, slug, postRelatedArticlesKey)

	return relatedArticles, inCache
}

func (c *redisCache) SetPostRelatedArticles(blogSlug, language, slug string, relatedArticles []goblogs.RelatedArticle) error {
	return c.set(relatedArticles, 0, blogSlug, language, slug, postRelatedArticlesKey)
}

func (c *redisCache) GetBlogsStatus() BlogsStatus {
	key := GetCacheKey(blogsStatusKey)

	value, _ := redisClient.Get(context.TODO(), key).Result()

	switch value {
	case string(rune(BlogsNotOK)):
		return BlogsNotOK

	case string(rune(BlogsOK)):
		return BlogsOK

	default:
		return BlogsOK
	}
}

func (c *redisCache) SetBlogsStatus(v BlogsStatus) error {
	key := GetCacheKey(blogsStatusKey)

	return c.Set(context.TODO(), key, string(rune(v)), 0).Err()
}

func (c *redisCache) Lock(key string) error {
	locker := redislock.New(redisClient)

	lock, err := locker.Obtain(key, 2*time.Minute, nil)

	if err != nil {
		return err
	}

	c.locks[key] = lock

	return nil
}

func (c *redisCache) Unlock(key string) error {
	err := c.locks[key].Release()

	if err != nil {
		return err
	}

	delete(c.locks, key)

	return nil
}
