package cache

import (
	"os"
	"time"

	"github.com/jasonlvhit/gocron"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/commerce/blogs_pumpkin/config"
	"a.yandex-team.ru/commerce/blogs_pumpkin/libs"
	"a.yandex-team.ru/commerce/blogs_pumpkin/libs/redislock"
	"a.yandex-team.ru/commerce/libs/goblogs"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
)

var (
	blogsForWarmUp []*goblogs.BlogIdentity
)

func init() {
	for _, slug := range config.Values.Blogs.RU {
		blogsForWarmUp = append(blogsForWarmUp, libs.Blogs.BlogIdentity(slug, "ru-RU"))
	}

	for _, slug := range config.Values.Blogs.EN {
		blogsForWarmUp = append(blogsForWarmUp, libs.Blogs.BlogIdentity(slug, "en-EN"))
	}

	for _, slug := range config.Values.Blogs.TR {
		blogsForWarmUp = append(blogsForWarmUp, libs.Blogs.BlogIdentity(slug, "tr-TR"))
	}
}

func WarmUpDefault() {
	if os.Getenv("FORCE_DEGRADE") == "1" {
		libs.Logger.Info("RUNNING IN FORCE_DEGRADE MODE")

		return
	}

	var err error

	libs.Logger.Info("starting warm-up the cache")

	for ok := false; !ok; ok = err == nil {
		err = WarmUp(DefaultCache, blogsForWarmUp...)

		if err != nil {
			libs.Logger.Error("can't warm-up the cache", log.Error(err))

			_ = DefaultCache.SetBlogsStatus(BlogsNotOK)

			time.Sleep(time.Duration(config.Values.Cache.WarmUpBackOff) * time.Second)
		} else {
			_ = DefaultCache.SetBlogsStatus(BlogsOK)
		}
	}
}

func RunScheduledDefaultWarmUp() {
	gocron.Every(config.Values.Cache.WarmUpPeriod).Seconds().Do(WarmUpDefault)

	<-gocron.Start()
}

func WarmUp(cache Cache, blogs ...*goblogs.BlogIdentity) (err error) {
	var eg errgroup.Group
	var alreadyStarted bool

	err = cache.Lock(config.Values.Cache.Version)

	if err != nil && err != redislock.ErrNotObtained {
		return err
	}

	for err == redislock.ErrNotObtained {
		libs.Logger.Info("warm-up already started, waiting")

		alreadyStarted = true

		time.Sleep(time.Duration(config.Values.Cache.WarmUpBackOff) * time.Second)

		err = cache.Lock(config.Values.Cache.Version)
	}

	if err != nil {
		return err
	}

	defer cache.Unlock(config.Values.Cache.Version)

	if alreadyStarted {
		return
	}

	sem := make(chan struct{}, config.Values.Cache.WarmUpConcurrencyLimit)

	for _, blog := range blogs {
		blog := blog

		eg.Go(func() error {
			return warmUpBlog(cache, blog, sem)
		})
	}

	err = eg.Wait()

	return
}

func warmUpBlog(cache Cache, blog *goblogs.BlogIdentity, sem chan struct{}) error {
	var eg errgroup.Group
	var hasRelatedArticles bool

	slug, language := blog.Slug, blog.Language

	sem <- struct{}{}

	eg.Go(func() error {
		defer func() { <-sem }()

		blogInfo, err := blog.GetInfo()

		if err != nil {
			return xerrors.Errorf("can't get %s_%s info: %w", slug, language, err)
		}

		hasRelatedArticles = blogInfo.HasRelatedArticles

		return cache.SetBlog(slug, language, blogInfo)
	})

	sem <- struct{}{}

	eg.Go(func() error {
		defer func() { <-sem }()

		categories, err := blog.GetAllCategories()

		if err != nil {
			return xerrors.Errorf("can't get %s_%s categories: %w", slug, language, err)
		}

		return cache.SetBlogCategories(slug, language, categories)
	})

	sem <- struct{}{}

	eg.Go(func() error {
		defer func() { <-sem }()

		posts, err := blog.GetAllPosts()

		if err != nil {
			return xerrors.Errorf("can't get %s_%s posts: %w", slug, language, err)
		}

		for _, post := range posts {
			post := post

			eg.Go(func() error {
				return warmUpPost(cache, blog, post, hasRelatedArticles, sem)
			})
		}

		return cache.SetBlogPosts(slug, language, posts)
	})

	eg.Go(func() error {
		tags, err := blog.GetAllTags()

		if err != nil {
			return err
		}

		return cache.SetBlogTags(slug, language, tags)
	})

	return eg.Wait()
}

func warmUpPost(cache Cache, blog *goblogs.BlogIdentity, post goblogs.Post, withRelatedArticles bool, sem chan struct{}) error {
	var eg errgroup.Group

	blogSlug, language, slug := blog.Slug, blog.Language, post.Slug

	sem <- struct{}{}

	eg.Go(func() error {
		defer func() { <-sem }()

		postInfo, err := blog.PostIdentity(post.ID).GetInfo()

		if err != nil {
			return xerrors.Errorf("can't get %s_%s post %s info: %w", blogSlug, language, slug, err)
		}

		return cache.SetPost(blogSlug, language, slug, postInfo)
	})

	sem <- struct{}{}

	eg.Go(func() error {
		defer func() { <-sem }()

		comments, err := blog.PostIdentity(post.ID).GetAllComments()

		if err != nil {
			return xerrors.Errorf("can't get %s_%s post %s comments: %w", blogSlug, language, slug, err)
		}

		return cache.SetPostComments(blogSlug, language, slug, comments)
	})

	if withRelatedArticles {
		sem <- struct{}{}

		eg.Go(func() error {
			defer func() { <-sem }()

			relatedArticles, err := blog.PostIdentity(post.ID).GetAllRelatedArticles()

			if err != nil {
				return xerrors.Errorf("can't get %s_%s post %s related articles: %w", blogSlug, language, slug, err)
			}

			return cache.SetPostRelatedArticles(blogSlug, language, slug, relatedArticles)
		})
	}

	return eg.Wait()
}
