package controllers

import (
	"math"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/commerce/blogs_pumpkin/context"
	"a.yandex-team.ru/commerce/libs/goblogs"
)

const (
	blogNotFoundMessage = `{"message":"blog not found","internalCode":"404_BNF"}`
)

func Blog(c echo.Context) error {
	cc := c.(context.Context)

	slug, language := getBlogParams(cc)

	if blog, ok := cc.Cache().GetBlog(slug, language); ok {
		return cc.JSON(http.StatusOK, blog)
	} else {
		return cc.String(http.StatusNotFound, blogNotFoundMessage)
	}
}

func BlogCategories(c echo.Context) error {
	cc := c.(context.Context)

	slug, language := getBlogParams(cc)

	if categories, ok := cc.Cache().GetBlogCategories(slug, language); ok {
		return cc.JSON(http.StatusOK, categories)
	} else {
		return cc.String(http.StatusNotFound, blogNotFoundMessage)
	}
}

func BlogTags(c echo.Context) error {
	cc := c.(context.Context)

	slug, language := getBlogParams(cc)

	if tags, ok := cc.Cache().GetBlogTags(slug, language); ok {
		return cc.JSON(http.StatusOK, tags)
	} else {
		return cc.String(http.StatusNotFound, blogNotFoundMessage)
	}
}

func BlogTag(c echo.Context) error {
	cc := c.(context.Context)

	blogSlug, language := getBlogParams(cc)
	tagSlug, err := url.PathUnescape(cc.Param("tag"))

	if err == nil {
		tags, _ := cc.Cache().GetBlogTags(blogSlug, language)

		for _, tag := range tags {
			if tag.Slug == tagSlug {
				return cc.JSON(http.StatusOK, tag)
			}
		}
	}

	return cc.String(http.StatusNotFound, blogNotFoundMessage)
}

func BlogPosts(c echo.Context) error {
	cc := c.(context.Context)

	slug, language := getBlogParams(cc)

	posts, ok := cc.Cache().GetBlogPosts(slug, language)

	if !ok {
		return cc.String(http.StatusNotFound, blogNotFoundMessage)
	}

	tags := getTagsQuery(cc)
	categoryID := cc.QueryParam("categoryId")
	fromPostID := cc.QueryParam("from")
	size := getSize(cc)

	var i int

	if fromPostID != "" {
		for i < len(posts) && posts[i].ID != fromPostID {
			i++
		}

		i++
	}

	result := make([]goblogs.Post, 0)

	for ; i < len(posts) && len(result) < size; i++ {
		if categoryID != "" && !contains(posts[i].CategoryIDs, categoryID) {
			continue
		}

		var postTags []string

		for _, postTag := range posts[i].Tags {
			postTags = append(postTags, postTag.Slug)
		}

		if !containsRight(postTags, tags) {
			continue
		}

		result = append(result, posts[i])
	}

	return cc.JSON(http.StatusOK, result)
}

func BlogPostsCustom(c echo.Context) error {
	return c.String(http.StatusOK, "[]")
}

func getBlogParams(cc context.Context) (slug, language string) {
	slug = cc.Param("blog")
	language = cc.QueryParam("lang")

	if language == "" {
		language = "ru-RU"
	}

	return
}

func getTagsQuery(cc context.Context) (tagSlugs []string) {
	tagParam := cc.QueryParam("tag")
	tagsParam := cc.QueryParam("tags")

	tags := append(strings.Split(tagsParam, ","), tagParam)

	for _, tag := range tags {
		if tag != "" {
			tagSlugs = append(tagSlugs, tag)
		}
	}

	return
}

func getSize(cc context.Context) (size int) {
	sizeParam := cc.QueryParam("size")

	size, _ = strconv.Atoi(sizeParam)

	if size == 0 {
		size = math.MaxInt32
	}

	return
}

func contains(s []string, e string) bool {
	for _, a := range s {
		if a == e {
			return true
		}
	}

	return false
}

func containsRight(s1 []string, s2 []string) bool {
	for _, e := range s2 {
		if !contains(s1, e) {
			return false
		}
	}

	return true
}
