const _ = require('lodash');

const BaseAccessControl = require('accessControl/base');
const assert = require('helpers/assert');
const Authority = require('models/authority');
const Lock = require('models/lock');

class AdminAccessControl extends BaseAccessControl {
    *hasAdminAccess(transaction) {
        this.authorizationRequired();

        const authority = yield Authority.find(this.uid, transaction);

        assert(authority.isAdmin, 403, 'User is not admin', 'NAD');
    }

    *hasAnalystAccess() {
        this.authorizationRequired();

        const authority = yield Authority.find(this.uid);

        assert(authority.isAnalyst, 403, 'User is not analyst', 'NAN');
    }

    *hasDeveloperAccess() {
        this.authorizationRequired();

        const authority = yield Authority.find(this.uid);

        assert(authority.isDeveloper, 403, 'User is not developer', 'UND', { uid: this.uid });
    }

    // eslint-disable-next-line complexity
    hasAnyAccess(authority) {
        const hasAnyAccess =
            authority.isDeveloper ||
            authority.isAnalyst ||
            authority.isAdmin ||
            authority.isAssessor ||
            authority.isEditor ||
            authority.isSupport;

        assert(hasAnyAccess, 403, 'User has no any access', 'NAA');
    }

    *hasAccessToLockExam(trialTemplateId, transaction) {
        const lock = yield Lock.tryFindLast(trialTemplateId, transaction);

        const login = _.get(lock, 'admin.login');
        const unlockDate = _.get(lock, 'unlockDate');
        const isAvailable = !_.isNull(unlockDate) || login === this.login;

        assert(isAvailable, 403, 'Exam is already being edited', 'EAE', { login });

        return unlockDate;
    }

    *hasSupportAccess(transaction) {
        this.authorizationRequired();

        const authority = yield Authority.find(this.uid, transaction);
        const hasAccess = authority.isAdmin || authority.isSupport;

        assert(hasAccess, 403, 'User has no support access', 'UNS');
    }

    *hasAccessToEditExam(transaction) {
        this.authorizationRequired();

        const authority = yield Authority.find(this.uid, transaction);
        const hasAccess = authority.isAdmin || authority.isEditor;

        assert(hasAccess, 403, 'User has no editor access', 'UEA');
    }

    *hasAccessToVideo(transaction) {
        this.authorizationRequired();

        const authority = yield Authority.find(this.uid, transaction);
        const hasAccess = authority.isAdmin || authority.isSupport || authority.isAssessor;

        assert(hasAccess, 403, 'User has no access to revise video', 'NRV');
    }
}

module.exports = AdminAccessControl;
