'use strict';

const BaseAccessControl = require('accessControl/base');

const assert = require('helpers/assert');

const Attempt = require('models/attempt');
const ProctoringResponseModel = require('models/proctoringResponse');
const userModels = require('models/user');

const _ = require('lodash');

class AttemptAccessControl extends BaseAccessControl {
    *hasAccessToCreateAttempt(examIdentity, transaction) {
        this.authorizationRequired();

        const attemptInfo = yield Attempt.getInfo(
            examIdentity,
            {
                uid: this.uid,
                authTypeCode: this.authType,
                login: this.login
            },
            transaction
        );

        const isAttemptStarted = attemptInfo.state === 'in_progress';

        assert(!isAttemptStarted, 403, 'Attempt is already started', 'AAS', { attemptId: attemptInfo.attemptId });

        const isAttemptAvailable = attemptInfo.state === 'enabled';

        assert(isAttemptAvailable, 403, 'Attempt does not available', 'ANA');
    }

    *hasAccessToQuestion(attempt) {
        yield this.hasAccessToAttempt(attempt);

        const isFinished = attempt.isFinished();

        assert(!isFinished, 403, 'Test finished', 'TFN');
    }

    *hasAccessToFinish(attempt, transaction) {
        yield this.hasAccessToAttempt(attempt, transaction);

        const expired = attempt.get('expired');

        assert(expired === 0, 403, 'Attempt have already finished', 'AAF');
    }

    *hasAccessToAttempt(attempt, transaction) {
        this.authorizationRequired();

        const userId = attempt.get('userId');

        yield this._checkUserId(userId, transaction);
    }

    *_checkUserId(attemptUserId, transaction) {
        const userModel = userModels(this.authType);
        const userData = yield userModel.findUser({ where: { uid: this.uid }, transaction });

        assert(userData, 404, 'User not found', 'UNF', { uid: this.uid, authType: this.authType });

        const userId = userData.get('id');

        assert(attemptUserId === userId, 403, 'Illegal user for attempt', 'IUA');
    }

    static hasAccessForRevision(responses, attemptId, passed) {
        assert(responses.length > 0, 403, 'Attempt does not have verdicts', 'AHV', { attemptId });

        const isRevisionWas = _.some(responses, 'isRevisionRequested');

        assert(!isRevisionWas, 403, 'Revision is already requested', 'RAR', { attemptId });

        const lastResponse = ProctoringResponseModel.findLastByResponses(responses);
        const isRevisionAvailable = passed && lastResponse.verdict === 'failed' &&
            (lastResponse.source === 'proctoring' || lastResponse.source === 'toloka');

        assert(isRevisionAvailable, 403, 'Revision is not available', 'RNA', { attemptId });
    }
}

module.exports = AttemptAccessControl;
