const os = require('os');
const { sqlLogger } = require('logger');

module.exports = {
    clientId: 'expert-api-testing',

    i18n: {
        languages: ['ru', 'en', 'tr', 'cn'],
        tankerLanguages: {
            ru: 'ru',
            en: 'en',
            tr: 'tr',
            cn: 'zh-Hans'
        },
        defaultLanguage: 'ru',
        tankerServicesToNullify: {
            direct: 'direct_nullify',
            'direct_en': 'direct_nullify',
            'direct_cn': 'direct_nullify',
            'direct_pro': 'direct_pro_nullify',
            metrika: 'metrika_nullify',
            'metrika_en': 'metrika_nullify',
            market: 'market_nullify',
            rsya: 'rsya_nullify',
            publisher: 'publisher_nullify',
            cpm: 'cpm_nullify',
            zen: 'zen_nullify'
        }
    },

    blackbox: {
        connection: {
            timeout: 500,
            retries: 5,
            family: 6,
            api: 'pass-test.yandex.ru'
        },

        /**
         * 27 - userinfo.firstname, если не null и непустое
         * 28 - userinfo.lastname, если не null и непустое
         * 1008 - accounts.login.uid (нормализованный логин пользователя)
         *
         * @see: https://wiki.yandex-team.ru/passport/dbmoving/#tipyatributov
         */
        attributes: '27,28,1008',
        tvmName: 'blackbox',
        chunkSize: 200
    },

    yandexTeamBlackbox: {
        connection: {
            timeout: 500,
            retries: 5,
            family: 6,
            api: 'blackbox.yandex-team.ru'
        },
        attributes: '1008',
        tvmName: 'yandexTeamBlackbox'
    },

    server: {
        port: process.env.DEPLOY_HTTP_PORT || process.env.PORT,
        hostname: os.hostname()
    },

    postgres: {
        uri: process.env.POSTGRES_URI || 'postgresql://localhost:5432/expert',
        options: {
            benchmark: true,
            dialect: 'postgres',
            pool: {
                max: 200
            },
            retry: {
                timeout: 5000,
                name: 'Query'
            },
            logging: sqlLogger
        }
    },

    mdsService: {
        write: {
            protocol: 'http',
            port: 1111
        },
        namespace: 'expert',
        maxRepeatCount: 3,
        token: process.env.MDS_TOKEN
    },

    avatarsService: {
        write: {
            protocol: 'http:',
            port: 13000
        },
        namespace: 'expert',
        maxRepeatCount: 3
    },

    tanker: {
        protocol: 'https',
        hostname: 'tanker-api.yandex-team.ru',
        port: 443,
        pathname: '/projects/export/json/',
        query: {
            'project-id': 'expert-v2',
            'branch-id': 'master',
            'keyset-id': 'cert',
            status: 'unapproved',
            'flat-keyset': 1
        }
    },

    token: 'A7bX4uf9cp',

    direct: {
        delay: 60 * 60 * 1000,
        sleep: 50,
        url: 'http://intapi.test.direct.yandex.ru:9000/UserRole'
    },

    mail: {
        smtp: 'outbound-relay.yandex.net'
    },

    mailLists: {
        agencies: {
            from: {
                name: 'Рассылка adv-expert-agency',
                address: 'adv-expert-agency@yandex-team.ru'
            },
            to: 'expert-dev-test@yandex-team.ru'
        },
        draftForModeration: {
            from: {
                name: 'Рассылка expert-dev',
                address: 'expert-dev@yandex-team.ru'
            },
            to: 'expert-dev-test@yandex-team.ru'
        },
        debug: 'expert-dev-test@yandex-team.ru'
    },

    freezing: {
        time: 3 * 60 * 60 * 1000,
        from: {
            name: 'Рассылка expert-dev',
            address: 'expert-dev@yandex-team.ru'
        },
        to: 'expert-dev-test@yandex-team.ru',
        accessLogins: [
            'yndx-anyok',
            'bloguser-05',
            'myshnorush',
            'ani.simanova'
        ]
    },

    // eslint-disable-next-line global-require
    draw: require('models/draw/data/options'),

    warnLogTimeout: 300,

    authToken: process.env.AUTH_TOKEN,

    yt: {
        host: 'http://hahn.yt.yandex.net',
        api: '/api/v3',
        authToken: process.env.YT_TOKEN,
        tableCapacity: 5,
        maxPendingTrialAge: 1000 * 60 * 60 * 24 * 2, // 2 days
        maxHeavyProxyAge: 1000 * 60, // 1 min
        inputTableSchema: [
            { name: 'trialId', type: 'int64' },
            { name: 'userPhoto', type: 'string' },
            { name: 'videos', type: 'any' },
            { name: 'isRevision', type: 'boolean' }
        ],
        clean: {
            dirPaths: [
                'archive/input',
                'archive/output',
                'input',
                'output',
                'logs'
            ],
            maxTTLMonths: 6
        },
        fail: {
            from: {
                name: 'Рассылка expert-dev',
                address: 'expert-dev@yandex-team.ru'
            },
            to: 'expert-dev-test@yandex-team.ru'
        }
    },

    s3: {
        accessKeyId: process.env.AWS_ACCESS_KEY_ID,
        secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
        endpoint: 'https://s3.mds.yandex.net',
        region: 'us-east-1',
        buckets: {
            public: 'expert',
            private: 'expert-hidden'
        },
        maxRepeatCount: 3,
        fail: {
            from: {
                name: 'Рассылка expert-dev',
                address: 'expert-dev@yandex-team.ru'
            },
            to: 'expert-dev-test@yandex-team.ru'
        },
        proxyPath: 'https://yastatic.net/s3',
        maxTrialsToUpload: 2,
        env: 'testing'
    },

    proctoring: {
        host: 'https://yandex-dev.proctoring.online',
        apiKey: process.env.PROCTORING_API_KEY,
        protocolPath: '/api/rest/v3/room',
        eventsPath: '/api/rest/v3/event',
        storagePath: '/api/rest/v3/storage',
        userPath: '/api/rest/v3/user',
        pendingRange: {
            from: parseInt(process.env.FROM_PENDING_RANGE, 10),
            to: parseInt(process.env.TO_PENDING_RANGE, 10)
        },
        maxRepeatCount: 3,
        fail: {
            from: {
                name: 'Рассылка expert-dev',
                address: 'expert-dev@yandex-team.ru'
            },
            to: 'expert-dev-test@yandex-team.ru'
        },
        maxLimitViolation: parseInt(process.env.MAX_LIMIT_VIOLATION, 10), // 0 - 100, в %
        minVideoDuration: parseInt(process.env.MIN_VIDEO_DURATION, 10), // в мс
        technicalMetrics: ['b2', 's2', 's1'],
        maxTechMetricsViolation: parseInt(process.env.MAX_TECH_METRICS_VIOLATION, 10), // 0 - 100, в %
        maxVideosDiff: 2500 // в мс
    },

    tvm: {
        host: `${process.env.DEPLOY_TVM_TOOL_URL}/tvm`,
        appId: process.env.TVM_ID,
        token: process.env.TVMTOOL_LOCAL_AUTHTOKEN,
        cache: {
            ticketMaxAge: 1000 * 60 * 9 // 9 min
        },
        headers: {
            Connection: 'Keep-Alive',
            Authorization: process.env.TVMTOOL_LOCAL_AUTHTOKEN,
            'Content-Type': 'application/json'
        },
        blackbox: {
            alias: 'blackbox',
            dsts: process.env.BLACKBOX_TVM_ID
        },
        yandexTeamBlackbox: {
            alias: 'blackbox-ya-team',
            dsts: process.env.BLACKBOX_YA_TEAM_TVM_ID
        },
        geoadv: {
            alias: 'geoadv-testing',
            dsts: 2000757
        },
        gozora: {
            alias: 'gozora',
            dsts: 2023123
        }
    },

    sender: {
        user: process.env.SENDER_USER
    },

    tldByLang: {
        ru: 'ru',
        en: 'com',
        cn: 'com'
    },

    frontHostPrefix: 'https://yandex.',

    frontExamPath: '/adv/expert/exam/',

    self: {
        protocol: 'https',
        basePathToFiles: 'v1/user/takeout'
    },

    minSpamKarma: parseInt(process.env.MIN_SPAM_KARMA, 10), // 0 - 100

    banDuration: { unit: 'year', count: 1 },

    fullFilledReports: ['bansByLogin'],

    certServices: [
        'metrika',
        'direct_pro'
    ],

    nullifiesByMetrics: {
        since: { unit: 'week', count: 1 },
        limit: 3
    },

    tracker: {
        options: {
            token: process.env.STARTREK_OAUTH_TOKEN,
            endpoint: 'https://st-api.test.yandex-team.ru/v2'
        },
        queue: 'PROCTORING',
        followers: [],
        fail: {
            from: {
                name: 'Рассылка expert-dev',
                address: 'expert-dev@yandex-team.ru'
            },
            to: 'expert-dev-test@yandex-team.ru'
        }
    },

    comdepReport: {
        from: new Date(2020, 0, 1), // 01.01.2020
        examSlugs: ['direct-pro'],
        dirPath: 'reports/comdep',
        tableName: 'trials',
        schema: [
            { name: 'login', type: 'string' },
            { name: 'firstname', type: 'string' },
            { name: 'lastname', type: 'string' },
            { name: 'email', type: 'string' },
            { name: 'trialId', type: 'int32' },
            { name: 'date', type: 'string' },
            { name: 'finalVerdict', type: 'string' },
            { name: 'certId', type: 'string' }
        ],
        chunkSize: 200000
    },

    geoadv: {
        exams: ['msp'],
        host: 'https://testing-api-geoadv.maps.yandex.net',
        path: '/v3/exam/status',
        fail: {
            from: {
                name: 'Рассылка expert-dev',
                address: 'expert-dev@yandex-team.ru'
            },
            to: 'expert-dev-test@yandex-team.ru'
        }
    }
};
