module.exports = function (sequelize, DataTypes) {
    const Trial = sequelize.define('trial', {
        id: {
            type: DataTypes.INTEGER,
            allowNull: false,
            autoIncrement: true,
            primaryKey: true
        },
        userId: {
            type: DataTypes.BIGINT,
            allowNull: false,
            field: 'user_id'
        },
        trialTemplateId: {
            type: DataTypes.INTEGER,
            allowNull: false,
            field: 'trial_template_id'
        },
        sequenceNumber: {
            type: DataTypes.INTEGER,
            allowNull: false,
            field: 'sequence_number'
        },
        started: {
            type: DataTypes.DATE,
            allowNull: true
        },
        finished: {
            type: DataTypes.DATE,
            allowNull: true
        },
        passed: {
            type: DataTypes.INTEGER,
            allowNull: false,
            defaultValue: 0
        },
        questionCount: {
            type: DataTypes.INTEGER,
            allowNull: false,
            field: 'question_count'
        },
        allowedFails: {
            type: DataTypes.INTEGER,
            allowNull: false,
            field: 'allowed_fails'
        },
        timeLimit: {
            type: DataTypes.INTEGER,
            allowNull: false,
            field: 'time_limit'
        },
        expired: {
            type: DataTypes.INTEGER,
            allowNull: false,
            defaultValue: 0
        },
        nullified: {
            type: DataTypes.INTEGER,
            allowNull: false,
            defaultValue: 0
        },
        nullifyReason: {
            type: DataTypes.ENUM('manual', 'metrics'),
            allowNull: true,
            field: 'nullify_reason'
        },
        openId: {
            type: DataTypes.STRING(36),
            allowNull: true,
            field: 'open_id'
        },
        pdf: {
            type: DataTypes.STRING(28),
            allowNull: true
        },
        filesStatus: {
            type: DataTypes.ENUM('initial', 'loading', 'saved', 'error'),
            allowNull: false,
            field: 'files_status',
            defaultValue: 'initial'
        },
        proctoringMetrics: {
            type: DataTypes.JSONB,
            allowNull: true,
            field: 'proctoring_metrics'
        }
    }, {
        timestamps: false,
        freezeTableName: true,
        tableName: 'trials',
        underscored: true,
        indexes: [
            {
                name: 'user_id_idx',
                fields: ['user_id']
            },
            {
                name: 'trial_template_id_idx',
                fields: ['trial_template_id']
            },
            {
                name: 'started_idx',
                fields: ['started']
            },
            {
                name: 'finished_idx',
                fields: ['finished']
            },
            {
                name: 'expired_idx',
                fields: ['expired']
            },
            {
                name: 'nullified_idx',
                fields: ['nullified']
            },
            {
                name: 'trials_open_id_unique_idx',
                unique: true,
                fields: ['open_id']
            },
            {
                name: 'trials_pdf_unique_idx',
                unique: true,
                fields: ['pdf']
            }
        ],
        classMethods: {
            createNextTrial: function createNextTrial(data) {
                return Trial
                    .findAll({
                        where: { userId: data.userId, trialTemplateId: data.trialTemplateId },
                        order: [['sequenceNumber', 'DESC']],
                        limit: 1,
                        transaction: data.transaction
                    })
                    .then(trials => trials.length ? trials[0].sequenceNumber + 1 : 1)
                    .then(sequenceNumber => {
                        data.sequenceNumber = sequenceNumber;
                        data.started = new Date();

                        return Trial.create(data, { transaction: data.transaction });
                    });
            }
        }
    });

    return Trial;
};
