const https = require('https');
const http = require('http');
const { URL } = require('url');

class HttpsProxyAgent extends https.Agent {
    constructor(options) {
        const { proxy, ...opts } = options;

        super(opts);

        this.proxy = typeof proxy === 'string'
            ? new URL(proxy)
            : proxy;
    }

    _getRequestOptions(options) {

        return {
            method: 'CONNECT',
            host: this.proxy.hostname,
            port: this.proxy.port,
            path: `${options.host}:${options.port}`,
            setHost: false,
            headers: {
                connection: this.keepAlive ? 'keep-alive' : 'close',
                host: `${options.host}:${options.port}`,
                ...options.headers
            },
            agent: false,
            timeout: options.timeout || 0
        };
    }

    createConnection(options, callback) {
        const requestOptions = this._getRequestOptions(options);

        // Necessary for the TLS check with the proxy to succeed.
        if (this.proxy.protocol === 'https:') {
            requestOptions.servername = this.proxy.hostname;
        }

        const request = (this.proxy.protocol === 'http:' ? http : https).request(requestOptions);

        request.once('connect', (response, socket) => {
            request.removeAllListeners();

            socket.removeAllListeners();

            if (response.statusCode === 200) {
                const secureSocket = super.createConnection({ ...options, socket });

                callback(null, secureSocket);
            } else {
                callback(new Error(`Bad response: ${response.statusCode}`), null);
            }
        });

        request.once('timeout', () => {
            request.destroy(new Error('Proxy timeout'));
        });

        request.once('error', err => {
            request.removeAllListeners();
            callback(err, null);
        });

        request.end();
    }
}

module.exports = HttpsProxyAgent;
