'use strict';

const db = require('db/postgres');
const _ = require('lodash');
const assert = require('helpers/assert');

function updateValues(model, values) {
    for (const key in values) {
        if (!Object.prototype.hasOwnProperty.call(values, key)) {
            continue;
        }

        model.set(key, values[key]);
    }

    return model;
}

function *upsert(schema, data, transaction) {
    const where = schema.getUniqueValue(data);
    const result = yield schema.model.findOrCreate({ where, transaction, defaults: data });
    const [storedData, created] = result;

    return created ?
        storedData :
        yield updateValues(storedData, data).save({ transaction });
}

function *findStoredQuestion(id, transaction) {
    const order = 'version DESC';
    const storedQuestion = yield db.Question.findById(id, { transaction, order });

    assert(storedQuestion, 404, 'Question not found', 'QNF', { id });

    return storedQuestion;
}

const schemas = [
    {
        name: 'sections',
        *getData(operations, data) {
            const exam = yield db.TrialTemplate.findById(data.examId);

            assert(exam, 404, 'Exam not found', 'ENF', { id: data.examId });

            return {
                serviceId: exam.serviceId,
                code: data.sectionsCode,
                title: data.sectionTitle
            };
        },
        getUniqueValue: entity => _.pick(entity, ['code', 'serviceId']),
        model: db.Section,
        key: 'sectionId',
        upsert
    },
    {
        name: 'categories',
        getData(operations, data) {
            return {
                id: data.categoryId,
                difficulty: 0,
                timeLimit: 0
            };
        },
        getUniqueValue: entity => _.pick(entity, ['id']),
        model: db.Category,
        key: 'categoryId',
        *upsert(schema, data, transaction) {
            const result = yield db.Category.findOrCreate({
                where: schema.getUniqueValue(data),
                defaults: data,
                transaction
            });

            return result[0];
        }
    },
    {
        name: 'trialTemplateAllowedFails',
        getData(operations, data) {
            const lastSection = _(operations.sections).last();

            return {
                trialTemplateId: data.examId,
                sectionId: lastSection,
                allowedFails: data.allowedFails
            };
        },
        getUniqueValue: entity => _.pick(entity, ['trialTemplateId', 'sectionId']),
        model: db.TrialTemplateAllowedFails,
        upsert
    },
    {
        name: 'trialTemplateToSections',
        getData(operations, data) {
            const lastSection = _(operations.sections).last();
            const lastCategory = _(operations.categories).last();

            return {
                trialTemplateId: data.examId,
                sectionId: lastSection,
                categoryId: lastCategory,
                quantity: data.quantity
            };
        },
        getUniqueValue: entity => _.pick(entity, ['trialTemplateId', 'sectionId', 'categoryId']),
        model: db.TrialTemplateToSection,
        upsert
    },
    {
        name: 'questions',
        *getData(operations, data) {
            const lastSection = _(operations.sections).last();
            const lastCategory = _(operations.categories).last();
            const type = /один/i.test(data.questionType) ? 0 : 1;

            if (data.questionId) {
                const question = yield db.Question.findById(data.questionId);

                assert(question, 404, 'Question not found', 'QNF', { id: data.questionId });
            }

            return {
                id: _.get(data, 'questionId', 0),
                active: _.get(data, 'questionActive', 1),
                sectionId: lastSection,
                categoryId: lastCategory,
                text: data.questionText,
                type
            };
        },
        getUniqueValue: entity => _.pick(entity, ['text', 'sectionId', 'categoryId']),
        model: db.Question,
        key: 'questionId',
        *upsert(schema, data, transaction) {
            if (!data.id) {
                return yield this._createNewQuestion(data, transaction);
            }

            const storedQuestion = yield findStoredQuestion(data.id, transaction);
            const hasChanged = _.some([
                'active',
                'text',
                'type'
            ], field => data[field] !== storedQuestion.get(field));

            if (!hasChanged) {
                this._setUpdatedQuestionInfo(storedQuestion, false);

                return storedQuestion;
            }

            storedQuestion.set('active', 0);
            yield storedQuestion.save({ transaction });

            if (!data.active) {
                this._setUpdatedQuestionInfo(storedQuestion, false);

                return storedQuestion;
            }

            return yield this._createNewQuestionVersion(data, storedQuestion, transaction);
        }
    },
    {
        name: 'answers',
        *getData(operations, data) {
            const lastQuestion = _(operations.questions).last();

            if (data.answerId) {
                const answer = yield db.Answer.findById(data.answerId);

                assert(answer, 404, 'Answer not found', 'ANF', { id: data.answerId });
            }

            return {
                id: _.get(data, 'answerId', 0),
                active: _.get(data, 'answerActive', 1),
                questionId: lastQuestion,
                correct: data.answerCorrect,
                text: data.answerText
            };
        },
        getUniqueValue: entity => _.pick(entity, ['text', 'questionId', 'questionVersion']),
        model: db.Answer
    }
];

class AdminBase {
    constructor() {
        this._updatedQuestions = {};
    }

    get schemas() {
        return schemas;
    }

    *_createNewQuestion(data, transaction) {
        const question = yield db.Question.create(data, { transaction });

        this._setUpdatedQuestionInfo(question, true);

        return question;
    }

    *_createNewQuestionVersion(data, storedQuestion, transaction) {
        data.version = storedQuestion.get('version') + 1;

        const question = yield db.Question
            .build(data)
            .save({ transaction });

        this._setUpdatedQuestionInfo(question, true);

        return question;
    }

    _setUpdatedQuestionInfo(question, isVersionUpdated) {
        const { id, version, active } = question;

        this._updatedQuestions[id] = {
            version,
            active,
            isVersionUpdated
        };
    }

}

module.exports = AdminBase;
