'use strict';

const _ = require('lodash');
const moment = require('moment');

const {
    TrialTemplate,
    Trial,
    Certificate,
    Role,
    User,
    Agency,
    AuthType
} = require('db/postgres');

class AgenciesLoader {
    static *_selectData(startTime) {
        const now = startTime ? new Date(startTime) : new Date();
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['id', 'title', 'isProctoring']
        };
        const includeRole = {
            model: Role,
            attributes: ['id', 'title']
        };
        const includeAgency = {
            model: Agency,
            attributes: ['login', 'manager']
        };
        const includeAuthType = {
            model: AuthType,
            where: { code: 'web' },
            attributes: ['id'],
            as: 'authType'
        };
        const includeUser = {
            model: User,
            attributes: ['id', 'login'],
            include: [includeRole, includeAgency, includeAuthType]
        };
        const includeTrial = {
            model: Trial,
            attributes: ['id', 'trialTemplateId'],
            where: { nullified: 0 },
            include: [includeTrialTemplate, includeUser]
        };

        return yield Certificate.findAll({
            attributes: ['id', 'confirmedDate', 'dueDate', 'firstname', 'lastname'],
            include: [includeTrial],
            where: { dueDate: { $gte: now }, active: 1 },
            order: [
                'trial.user.agency.login',
                'trial.user.login',
                'id'
            ],
            raw: true
        });
    }

    static *getFlatData(startTime) {
        const data = yield AgenciesLoader._selectData(startTime);

        return data
            .map(certData => {
                return {
                    agencyLogin: _.get(certData, 'trial.user.agency.login'),
                    managerLogin: _.get(certData, 'trial.user.agency.manager'),
                    userLogin: _.get(certData, 'trial.user.login', ''),
                    firstname: _.get(certData, 'firstname', ''),
                    lastname: _.get(certData, 'lastname', ''),
                    userRole: _.get(certData, 'trial.user.role.title'),
                    examTitle: _.get(certData, 'trial.trialTemplate.title'),
                    isProctoring: _.get(certData, 'trial.trialTemplate.isProctoring') ? 1 : 0,
                    certId: certData.id.toString(),
                    confirmedDate: certData.confirmedDate ? moment(certData.confirmedDate).format('DD.MM.YYYY') : '-',
                    dueDate: moment(certData.dueDate).format('DD.MM.YYYY')
                };
            })
            .filter(info => info.agencyLogin);
    }
}

module.exports = AgenciesLoader;
