'use strict';

const assert = require('helpers/assert');
const dateHelper = require('helpers/dateHelper');
const getBanData = require('helpers/getBanData');

const Base = require('models/base');
const userModels = require('models/user');
const CertificateModel = require('models/certificate');
const ExamModel = require('models/exam');
const MdsModel = require('models/mds');
const ProctoringResponseModel = require('models/proctoringResponse');
const ProctorEdu = require('models/proctoring/proctorEdu');

const _ = require('lodash');
const moment = require('moment');
const config = require('yandex-config');

const {
    AuthType,
    Certificate,
    Freezing,
    ProctoringResponses,
    Question,
    Section,
    Service,
    Trial,
    TrialTemplate,
    TrialTemplateAllowedFails,
    TrialTemplateToSection,
    TrialToQuestion,
    Type,
    User
} = require('db/postgres');

class Attempt extends Base {
    get openId() {
        return this._data.openId;
    }

    get pdf() {
        return this._data.pdf;
    }

    // eslint-disable-next-line max-statements
    static *create(data, state, authType) {
        const transaction = _.get(state, 'transaction');
        const exam = yield Attempt._findExam(data.examIdentity, transaction);

        assert(!exam.isProctoring || data.openId, 400, 'Open id is absent', 'OIA');

        let questionsCount = 0;
        const questions = yield exam.trialTemplateToSection.map(function *(section) {
            questionsCount += section.quantity;

            const sectionQuestions = yield Question.findAll({
                where: {
                    sectionId: section.sectionId,
                    categoryId: section.categoryId,
                    active: 1
                },
                attributes: ['id', 'version'],
                transaction
            });

            return _(sectionQuestions)
                .sampleSize(section.quantity)
                .value();
        });

        const userModel = userModels(authType);
        const user = yield userModel.findAndRenew(state.user, transaction);
        let trial;

        try {
            trial = yield Trial.createNextTrial({
                userId: user.id,
                trialTemplateId: exam.id,
                allowedFails: _.sumBy(exam.trialTemplateAllowedFails, 'allowedFails'),
                questionCount: questionsCount,
                timeLimit: exam.timeLimit,
                openId: data.openId,
                transaction
            });
        } catch (error) {
            const isOtherError = error.name !== 'SequelizeUniqueConstraintError';
            const details = _.get(error, 'errors.0');

            assert(isOtherError, 400, 'Unique constraint error', 'UCE', { details });

            throw error;
        }

        const insertData = _(questions)
            .flatten()
            .map((question, i) => {
                return {
                    trialId: trial.id,
                    questionId: question.id,
                    questionVersion: question.version,
                    seq: i + 1
                };
            })
            .value();

        yield TrialToQuestion.bulkCreate(insertData, { transaction });

        trial.examSlug = exam.slug;

        return new Attempt(trial);
    }

    static *_findExam(examIdentity, transaction) {
        const condition = ExamModel.getFindCondition(examIdentity);
        const includeSections = {
            model: TrialTemplateToSection,
            as: 'trialTemplateToSection',
            attributes: ['quantity', 'sectionId', 'categoryId']
        };
        const includeFails = {
            model: TrialTemplateAllowedFails,
            as: 'trialTemplateAllowedFails',
            attributes: ['allowedFails']
        };
        const exam = yield TrialTemplate.findOne({
            where: condition,
            include: [includeSections, includeFails],
            attributes: ['id', 'timeLimit', 'slug', 'isProctoring'],
            transaction
        });

        assert(exam, 404, 'Exam not found', 'ENF');

        return exam;
    }

    // eslint-disable-next-line complexity
    static *getInfo(examIdentity, userData, transaction) {
        const {
            isBannedOnTest,
            isSuperBanned,
            actualLogin,
            expiredDate
        } = yield getBanData(examIdentity, userData, transaction);

        if (isSuperBanned || isBannedOnTest) {
            return {
                state: 'banned',
                hasValidCert: false,
                expiredDate
            };
        }

        const exam = yield Attempt._getExam(examIdentity, transaction);
        const trials = yield Attempt._getUserTrials(exam.id, userData, transaction);

        exam.trials = trials;
        const attemptState = yield Attempt._getState(exam, userData.login, transaction);

        if (attemptState.state === 'enabled' && actualLogin && actualLogin !== userData.login) {
            return { state: 'notActualLogin' };
        }

        const now = new Date();
        const hasValidCert = _.some(trials, trialData => {
            const dueDate = _.get(trialData, 'certificates.0.dueDate');
            const isActive = _.get(trialData, 'certificates.0.active');

            return dueDate > now && isActive;
        });

        return _.assign(attemptState, { hasValidCert });
    }

    static *_getExam(examIdentity, transaction) {
        const exam = yield TrialTemplate.findOne({
            where: ExamModel.getFindCondition(examIdentity),
            attributes: [
                'id',
                'timeLimit',
                'periodBeforeCertificateReset',
                'delays',
                'allowedTriesCount',
                'isProctoring'
            ],
            transaction
        });

        assert(exam, 404, 'Exam not found', 'ENF');

        return exam.toJSON();
    }

    static *_getUserTrials(examId, userData, transaction) {
        const includeAuthType = {
            model: AuthType,
            where: { code: userData.authTypeCode },
            attributes: ['id'],
            as: 'authType'
        };
        const includeUser = {
            model: User,
            where: { uid: userData.uid },
            attributes: ['id', 'uid'],
            include: includeAuthType
        };
        const includeCertificates = {
            model: Certificate,
            as: 'certificates',
            required: false,
            attributes: ['active', 'dueDate']
        };

        return yield Trial.findAll({
            where: { nullified: 0, trialTemplateId: examId },
            attributes: ['started', 'expired', 'id', 'finished', 'passed', 'openId'],
            include: [includeCertificates, includeUser],
            order: [['started', 'DESC']],
            transaction
        });
    }

    // eslint-disable-next-line complexity
    static *_defineState(exam, transaction) {
        const [trial] = exam.trials;

        if (!trial) {
            return { state: 'enabled' };
        }

        if (!this._isFinished(trial, exam)) {
            return {
                state: 'in_progress',
                attemptId: trial.id,
                openId: trial.openId
            };
        }

        const proctoringStatus = yield Attempt._defineProctoringStatus(trial.id, transaction);

        if (trial.passed && proctoringStatus && proctoringStatus.isPending) {
            return { state: 'pending' };
        }

        const trialNumber = yield Attempt._getTrialNumber(exam, transaction);

        if (trialNumber + 1 >= exam.allowedTriesCount) {
            return { state: 'disabled' };
        }

        return Attempt._defineStateByOldTrials(exam, trialNumber, _.get(proctoringStatus, 'verdictTime'));
    }

    static *_defineProctoringStatus(trialId, transaction) {
        const proctoringResponses = yield ProctoringResponseModel.findByTrialIds([trialId], transaction);
        const lastResponse = ProctoringResponseModel.findLastByResponses(proctoringResponses);

        if (!lastResponse) {
            return;
        }

        const isPending = lastResponse.verdict === 'pending';
        const firstFailed = _.find(proctoringResponses, { verdict: 'failed' });
        const lastSuccess = _.findLast(proctoringResponses, { verdict: 'correct' });

        let verdictTime;

        if (lastSuccess) {
            verdictTime = lastSuccess.time;
        } else if (firstFailed) {
            verdictTime = firstFailed.time;
        } else {
            verdictTime = lastResponse.time;
        }

        return { isPending, verdictTime };
    }

    static _defineStateByOldTrials(exam, trialNumber, verdictTime) {
        const dateReset =
            Attempt._getDateResetCert(exam) ||
            Attempt._getDateResetAttempt(exam, trialNumber, verdictTime);

        return dateReset ? {
            state: 'disabled',
            availabilityDate: dateReset
        } : { state: 'enabled' };
    }

    static *_getState(exam, login, transaction) {
        const attemptState = yield Attempt._defineState(exam, transaction);
        const hasFrozenAccess = _.includes(config.freezing.accessLogins, login);

        if (attemptState.state !== 'enabled' || hasFrozenAccess) {
            return attemptState;
        }

        const freezingData = yield Freezing.findOne({
            where: { trialTemplateId: exam.id },
            order: [['startTime', 'DESC']],
            transaction
        });

        if (freezingData) {
            const freezeFinishTime = freezingData.get('finishTime');

            if (new Date() < freezeFinishTime) {
                return {
                    state: 'frozen',
                    freezeFinishTime: moment(freezeFinishTime).toISOString()
                };
            }
        }

        return attemptState;
    }

    static _isFinished(trial, exam) {
        const deadLine = moment(trial.started).add(exam.timeLimit);

        return Boolean(trial.expired || deadLine < new Date());
    }

    static _getDateResetCert(exam) {
        const certDueDate = _.get(exam, 'trials.0.certificates.0.dueDate');

        if (!certDueDate) {
            return;
        }

        const isActive = _.get(exam, 'trials.0.certificates.0.active');
        let resetDate = null;

        if (isActive) {
            const delta = exam.periodBeforeCertificateReset;

            resetDate = dateHelper.subtractFromDate(certDueDate, delta);
        } else {
            resetDate = moment(certDueDate).startOf('day').toDate();
        }

        if (resetDate > new Date()) {
            return resetDate;
        }
    }

    static *_getTrialNumber(exam, transaction) {
        let trialNumber = 0;

        for (const trial of exam.trials) {
            const isTrialPassed = yield Attempt._isTrialPassed(exam.isProctoring, trial, transaction);

            if (isTrialPassed) {
                break;
            }

            trialNumber += 1;
        }

        return trialNumber;
    }

    static *_isTrialPassed(isProctoring, trial, transaction) {
        if (!trial.passed) {
            return false;
        }

        if (!isProctoring) {
            return true;
        }

        const proctoringResponse = yield ProctoringResponseModel.tryFindLast(trial.id, transaction);

        return proctoringResponse.isCorrect();
    }

    // eslint-disable-next-line complexity
    static _getDateResetAttempt(exam, trialNumber, verdictTime) {
        const { delays, trials: [trial] } = exam;

        if (!delays.length || !trial) {
            return;
        }

        const delay = delays[trialNumber - 1] || _.last(delays);
        const trialDate = verdictTime || trial.started;
        const dateResetDelay = dateHelper.addToDate(trialDate, delay);

        if (dateResetDelay > new Date()) {
            return dateResetDelay;
        }
    }

    get fields() {
        return [
            'id',
            'userId',
            'trialTemplateId',
            'started',
            'finished',
            'passed',
            'questionCount',
            'allowedFails',
            'timeLimit',
            'expired',
            'nullified',
            'nullifyReason',
            'examSlug',
            'openId',
            'pdf'
        ];
    }

    static *findById(id, transaction) {
        const attempt = yield Trial.findById(id, { transaction });

        assert(attempt, 404, 'Attempt not found', 'ATF');

        return new Attempt(attempt);
    }

    isFinished() {
        const deadline = new Date(this.get('started').getTime() + this.get('timeLimit'));

        return Boolean(this.get('expired') || deadline < new Date());
    }

    *finish(isCritMetrics, transaction) {
        const attemptId = this.get('id');
        const passed = yield Attempt._isAttemptPassed(attemptId, transaction);

        yield this._complete(isCritMetrics, passed, transaction);
    }

    static *_isAttemptPassed(trialId, transaction) {
        const sections = yield Attempt._getSectionsInfo(trialId, transaction);

        for (const section of sections) {
            if (section.passed === 0) {
                return 0;
            }
        }

        return 1;
    }

    static *_getSectionsInfo(trialId, transaction) {
        const includeTrial = {
            model: Trial,
            attributes: [],
            where: { id: trialId }
        };
        const includeTrialsToQuestion = {
            model: TrialToQuestion,
            as: 'trialToQuestions',
            where: { trialId },
            attributes: ['questionId', 'correct']
        };
        const condition = yield Attempt.getQuestionsCondition(trialId, transaction);
        const includeQuestions = {
            model: Question,
            where: condition,
            as: 'questions',
            attributes: ['id'],
            include: includeTrialsToQuestion
        };
        const includeFails = {
            model: TrialTemplateAllowedFails,
            as: 'trialTemplateAllowedFails',
            attributes: ['sectionId', 'allowedFails']
        };

        const queries = yield [
            Section.findAll({
                attributes: ['id', 'code', 'title'],
                include: includeQuestions,
                order: [[includeQuestions, includeTrialsToQuestion, 'seq', 'ASC']],
                transaction
            }),
            TrialTemplate.findOne({
                include: [includeTrial, includeFails],
                attributes: [],
                transaction
            })
        ];

        const exam = queries[1].toJSON();
        const allowedFailsBySections = _(exam.trialTemplateAllowedFails)
            .groupBy('sectionId')
            .mapValues(section => _.sumBy(section, 'allowedFails'))
            .value();

        return queries[0].map(section => {
            const correctCount = _.sumBy(section.questions, 'trialToQuestions.0.correct');
            const totalCount = section.questions.length;
            const allowedFails = _.get(allowedFailsBySections, section.id, 0);
            const passed = correctCount >= totalCount - allowedFails ? 1 : 0;

            return _.assign(
                _.pick(section, ['code', 'title']),
                { passed, correctCount, totalCount, allowedFails }
            );
        });
    }

    static *getQuestionsCondition(trialId, transaction) {
        const includeTrialsToQuestion = {
            model: TrialToQuestion,
            as: 'trialToQuestions',
            where: { trialId },
            attributes: []
        };

        const questions = yield Question.findAll({
            attributes: ['id', 'version'],
            include: includeTrialsToQuestion,
            order: [
                [includeTrialsToQuestion, 'seq'],
                ['version', 'DESC']
            ],
            transaction
        });

        const $or = _(questions)
            .uniqBy('id')
            .map(question => question.toJSON())
            .value();

        return { $or };
    }

    *getResult(userData, transaction) {
        const sections = yield Attempt._getSectionsInfo(this.id, transaction);
        const total = Attempt._getTotalInfo(sections);

        if (this.get('expired') === 0) {
            yield this._complete(false, total.passed, transaction);
        }

        const proctoringData = yield Attempt._getProctoringStatus(this.id, transaction);

        _.assign(total, proctoringData);

        const examData = yield Attempt._getExamInfo(this.id, transaction);
        const exam = examData.toJSON();

        const resultInfo = _.assign(
            { total, sections },
            {
                service: exam.service,
                type: exam.type,
                finished: _.get(exam, 'trials.0.finished'),
                isNullified: Boolean(_.get(exam, 'trials.0.nullified')),
                isProctoring: exam.isProctoring
            }
        );

        if (_.get(exam, 'trials.0.certificates').length) {
            Attempt.setCertificatePaths(exam);
            _.assign(resultInfo, _.get(exam, 'trials.0.certificates.0'));
        }

        const attemptInfo = yield Attempt.getInfo(
            this.get('trialTemplateId'),
            {
                uid: userData.uid,
                login: userData.login,
                authTypeCode: userData.authType
            },
            transaction
        );

        if (attemptInfo.availabilityDate) {
            resultInfo.availabilityDate = attemptInfo.availabilityDate;
        }

        return resultInfo;
    }

    static *_getProctoringStatus(trialId, transaction) {
        const proctoringResponses = yield ProctoringResponseModel.findByTrialIds([trialId], transaction);
        const isRevisionRequested = _.some(proctoringResponses, 'isRevisionRequested');
        const lastResponse = ProctoringResponseModel.findLastByResponses(proctoringResponses);

        return lastResponse ? {
            lastVerdict: lastResponse.verdict,
            lastSource: lastResponse.source,
            isRevisionRequested
        } : {};
    }

    static *_isTestWithProctoring(trialId) {
        const includeTrial = {
            model: Trial,
            where: { id: trialId },
            attributes: []
        };
        const exam = yield TrialTemplate.findOne({
            include: includeTrial,
            attributes: ['isProctoring'],
            raw: true
        });

        return exam.isProctoring;
    }

    static *getProctoringAnswer(openId) {
        const sessionData = yield ProctorEdu.getSessionData(openId);
        const evaluation = _.get(sessionData, 'evaluation', null);
        const { pendingRange: { from, to } } = config.proctoring;
        const isPending = evaluation >= from && evaluation < to;
        let verdict = null;

        if (_.isNull(evaluation) || isPending) {
            verdict = 'pending';
        } else if (evaluation >= to) {
            verdict = 'correct';
        } else {
            verdict = 'failed';
        }

        return {
            response: { source: 'proctoring', verdict, evaluation },
            sessionData
        };
    }

    *_complete(isCritMetrics, passed, transaction) {
        const trialId = this.id;

        this.set('expired', 1);
        this.set('finished', new Date());
        this.set('passed', passed);

        const isProctoring = yield Attempt._isTestWithProctoring(trialId);
        const data = { isProctoring };

        if (isProctoring) {
            const proctoringResponse = yield this.processProctoringAnswer(isCritMetrics, transaction);

            data.isProctoringCorrect = proctoringResponse.isCorrect();
        }

        yield this.save({ transaction });

        yield CertificateModel.tryCreate(this, data, transaction);
    }

    *processProctoringAnswer(isCritMetrics, transaction) {
        const { response, sessionData } = yield Attempt.getProctoringAnswer(this.openId);
        const isMetricsHigh = ProctorEdu.isMetricsHigh(sessionData);

        const trialId = this.id;
        const proctoringDataArray = [_.pick(response, ['source', 'verdict', 'evaluation'])];

        if (isMetricsHigh) {
            proctoringDataArray.push({ source: 'metrics', verdict: 'failed' });
        }

        if (isCritMetrics) {
            proctoringDataArray.push({ source: 'crit-metrics', verdict: 'failed' });
        }

        return yield ProctoringResponseModel.create(trialId, proctoringDataArray, transaction);
    }

    static *isNullifiesByMetricsLimitExceeded(userId, trialTemplateId) {
        const { limit, since } = config.nullifiesByMetrics;

        const nullifiesCount = yield Trial.count({
            where: {
                userId,
                trialTemplateId,
                nullifyReason: 'metrics',
                started: {
                    $gte: moment().subtract(since.count, since.unit).toDate()
                }
            }
        });

        return nullifiesCount >= limit;
    }

    static *nullify(trialIds, nullifyReason, transaction) {
        return yield Trial.update({ nullified: 1, nullifyReason }, {
            fields: ['nullified', 'nullifyReason'],
            where: { id: trialIds },
            transaction
        });
    }

    static _getTotalInfo(sections) {
        const total = { totalCount: 0, correctCount: 0, passed: 1 };

        return sections.reduce((totalInfo, section) => {
            totalInfo.totalCount += section.totalCount;
            totalInfo.correctCount += section.correctCount;
            if (section.passed === 0) {
                totalInfo.passed = 0;
            }

            return totalInfo;
        }, total);
    }

    static setCertificatePaths(trialTemplate) {
        const imagePath = _.get(trialTemplate, 'trials.0.certificates.0.imagePath');

        _(trialTemplate)
            .set('trials.0.certificates.0.imagePath', MdsModel.getAvatarsPath(imagePath))
            .value();
    }

    static *_getExamInfo(trialId, transaction) {
        const includeCertificates = {
            model: Certificate,
            as: 'certificates',
            attributes: [
                ['id', 'certId'],
                'firstname',
                'lastname',
                'dueDate',
                'active',
                'imagePath'
            ]
        };
        const includeServices = {
            model: Service,
            as: 'service'
        };
        const includeTypes = {
            model: Type,
            as: 'type'
        };
        const includeTrial = {
            model: Trial,
            attributes: ['id', 'finished', 'started', 'expired', 'nullified'],
            where: { id: trialId },
            include: includeCertificates
        };

        return yield TrialTemplate.findOne({
            include: [includeTypes, includeServices, includeTrial],
            transaction
        });
    }

    static *getLastFailedTrials(examSlugs, userData) {
        const includeAuthType = {
            model: AuthType,
            where: { code: userData.authTypeCode },
            attributes: [],
            as: 'authType'
        };
        const includeUser = {
            model: User,
            where: { uid: userData.uid },
            attributes: [],
            include: includeAuthType
        };
        const includeTrials = {
            model: Trial,
            where: { nullified: 0 },
            attributes: ['id', 'passed', 'started'],
            include: includeUser
        };

        const exams = yield TrialTemplate.findAll({
            where: { slug: { $in: examSlugs } },
            attributes: ['slug', 'isProctoring'],
            include: includeTrials,
            order: [
                'slug',
                [includeTrials, 'started', 'DESC']
            ]
        });

        const failedExams = yield _(exams)
            .map(function *(exam) {
                const trial = _.get(exam, 'trials.0', {});
                const { passed } = trial;
                const proctoringResponse = yield ProctoringResponseModel.tryFindLast(trial.id);

                if (passed && (!exam.isProctoring || !proctoringResponse.isFailed())) {
                    return;
                }

                const attemptInfo = yield Attempt.getInfo(exam.slug, userData);

                if (attemptInfo.state === 'in_progress') {
                    return;
                }

                const sections = yield Attempt._getSectionsInfo(trial.id);

                return _.assign(
                    {
                        trialId: trial.id,
                        examSlug: exam.slug,
                        started: trial.started,
                        source: _.get(proctoringResponse, 'source'),
                        passed,
                        sections
                    },
                    _.pick(attemptInfo, ['state', 'availabilityDate'])
                );
            }, this)
            .compact()
            .value();

        return _.compact(failedExams);
    }

    static *getPendingTrials(examSlugs, userData) {
        const includeAuthType = {
            model: AuthType,
            where: { code: userData.authTypeCode },
            attributes: [],
            as: 'authType'
        };
        const includeUser = {
            model: User,
            where: { uid: userData.uid },
            attributes: [],
            include: includeAuthType
        };
        const includeTrials = {
            model: Trial,
            where: { nullified: 0, passed: 1 },
            attributes: ['id', 'started'],
            include: includeUser
        };

        const proctoringExams = yield TrialTemplate.findAll({
            where: {
                slug: { $in: examSlugs },
                isProctoring: true
            },
            attributes: ['slug'],
            include: includeTrials,
            order: [
                'slug',
                [includeTrials, 'started', 'DESC']
            ]
        });

        const pendingExams = yield proctoringExams.map(function *(exam) {
            const attemptInfo = yield Attempt.getInfo(exam.slug, userData);

            if (attemptInfo.state !== 'pending') {
                return;
            }

            return {
                examSlug: exam.slug,
                trialId: _.get(exam, 'trials.0.id'),
                started: _.get(exam, 'trials.0.started')
            };
        });

        return _.compact(pendingExams);
    }

    static *getNewPendingTrials(transaction) {
        const includeProctoringResponses = {
            model: ProctoringResponses,
            where: {
                source: 'proctoring',
                verdict: 'pending',
                isSentToToloka: false,
                isLast: true
            },
            attributes: [],
            as: 'proctoringResponses'
        };

        const includeUser = {
            model: User,
            attributes: ['uid']
        };

        const trials = yield Trial.findAll({
            where: {
                nullified: 0,
                passed: 1,
                filesStatus: 'saved'
            },
            attributes: ['id', 'userId', 'started', 'openId'],
            include: [includeProctoringResponses, includeUser],
            order: [['started', 'ASC']],
            transaction
        });

        return trials.map(trial => ({
            id: trial.id,
            userId: trial.userId,
            userUid: trial.user.uid,
            started: trial.started,
            openId: trial.openId,
            isRevision: false
        }));
    }

    static *getTrialsWithInitialFilesStatus() {
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            where: { isProctoring: true },
            attributes: []
        };
        const trials = yield Trial.findAll({
            where: {
                filesStatus: 'initial',
                expired: 1
            },
            attributes: ['id', 'openId'],
            include: includeTrialTemplate,
            order: [['id']],
            raw: true,
            limit: config.s3.maxTrialsToUpload
        });

        return trials.map(trial => _.pick(trial, ['id', 'openId']));
    }

    static *setFilesStatusAndPdf(openId, data, transaction) {
        const { pdf, filesStatus } = data;

        yield Trial.update(
            { filesStatus, pdf },
            {
                where: { openId },
                fields: ['filesStatus', 'pdf'],
                transaction
            }
        );
    }

    static isExpiredTrialExist(pendingTrials) {
        const { maxPendingTrialAge } = config.yt;
        const expiredTime = Date.now() - maxPendingTrialAge;

        const expiredTrials = pendingTrials.filter(trial => {
            const startDate = moment(trial.started).toDate();

            return startDate.getTime() < expiredTime;
        });

        return !_.isEmpty(expiredTrials);
    }

    static *findByIds(ids, attributes, transaction) {
        return yield Trial.findAll({
            where: { id: ids },
            raw: true,
            attributes,
            transaction
        });
    }

    static *tryFindCertAndAppeal(trialId) {
        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: ['id'],
            required: false
        };
        const includeResponse = {
            model: ProctoringResponses,
            as: 'proctoringResponses',
            where: { source: 'appeal', isLast: true },
            attributes: ['verdict'],
            required: false
        };
        const trial = yield Trial.findOne({
            where: { id: trialId },
            attributes: [],
            include: [includeCertificate, includeResponse]
        });

        return {
            verdict: _.get(trial, 'proctoringResponses.0.verdict'),
            certId: _.get(trial, 'certificates.0.id')
        };
    }

    static *getTrialsInfo(ids, transaction) {
        const includeUser = {
            model: User,
            attributes: ['login']
        };

        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['slug']
        };

        const trials = yield Trial.findAll({
            where: { id: ids },
            attributes: ['id', 'started', 'nullified'],
            order: [['id']],
            include: [includeUser, includeTrialTemplate],
            transaction
        });

        return trials.map(trial => {
            const { id, started, nullified } = trial;

            return {
                trialId: id,
                started,
                nullified,
                login: _.get(trial, 'user.login'),
                trialTemplateSlug: _.get(trial, 'trialTemplate.slug')
            };
        });
    }

    static *getNewTrialsForRevision(transaction) {
        const includeProctoringResponses = {
            model: ProctoringResponses,
            where: {
                isSentToToloka: false,
                isRevisionRequested: true,
                isLast: true
            },
            attributes: [],
            as: 'proctoringResponses'
        };

        const trials = yield Trial.findAll({
            where: {
                nullified: 0,
                passed: 1,
                filesStatus: 'saved'
            },
            attributes: ['id', 'userId', 'started', 'openId'],
            include: [includeProctoringResponses],
            order: [['started', 'ASC']],
            transaction
        });

        return trials.map(trial => ({
            id: trial.id,
            userId: trial.userId,
            started: trial.started,
            openId: trial.openId,
            isRevision: true
        }));
    }

    static *saveProctoringMetrics(id, proctoringMetrics, transaction) {
        yield Trial.update({ proctoringMetrics }, {
            fields: ['proctoringMetrics'],
            where: { id },
            transaction
        });
    }

    static *getProctoringMetrics(id, transaction) {
        const trial = yield Trial.findOne({
            where: { id },
            attributes: ['proctoringMetrics'],
            transaction
        });

        return trial.proctoringMetrics;
    }

    static *getAttemptInfo(trialId, transaction) {
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['slug']
        };
        const includeCertificate = {
            model: Certificate,
            as: 'certificates',
            attributes: ['id']
        };

        return yield Trial.findOne({
            where: { id: trialId },
            attributes: ['id'],
            include: [includeCertificate, includeTrialTemplate],
            transaction
        });
    }
}

module.exports = Attempt;
