'use strict';

const Base = require('models/base');

const assert = require('helpers/assert');
const assertIsNumber = assert.isNumber;
const _ = require('lodash');

const {
    Answer,
    Section,
    Question,
    TrialToQuestion,
    Trial,
    TrialTemplate
} = require('db/postgres');

/**
 * @see: https://github.yandex-team.ru/expert/expert/blob/a07b88bf60644c146b5989c2a7030167755cd5fd/src/main/java/ru/yandex/expert/model/CourseQuestionType.java#L11-L12
 */
const ANSWER_STATES = [
    'not_answered',
    'answered',
    'skipped'
];
const ANSWER_CODES = ANSWER_STATES.reduce((result, state, index) => {
    result[state] = index;

    return result;
}, {});

class AttemptQuestion extends Base {
    static *findByNumber(trialId, seq) {
        const options = { where: { trialId, seq } };
        const attemptQuestion = yield AttemptQuestion._findQuestion(options);

        assert(attemptQuestion, 404, 'Attempt question not found', 'QNF');

        const countAnswered = yield AttemptQuestion._getAnsweredCount(trialId);

        return new AttemptQuestion({ attemptQuestion, countAnswered });
    }

    static *findNext(trialId) {
        const options = {
            where: { trialId, answered: { $ne: 1 } },
            order: [
                ['answered', 'ASC'],
                ['seq', 'ASC']
            ]
        };
        const attemptQuestion = yield AttemptQuestion._findQuestion(options);
        const countAnswered = yield AttemptQuestion._getAnsweredCount(trialId);

        return new AttemptQuestion({ attemptQuestion, countAnswered });
    }

    static *_getAnsweredCount(trialId) {
        return yield TrialToQuestion.count({
            where: { answered: 1, trialId }
        });
    }

    static *_findQuestion(options) {
        const includeAnswers = {
            model: Answer,
            as: 'answers',
            required: false
        };
        const includeQuestions = {
            model: Question,
            as: 'question',
            include: [includeAnswers, Section]
        };
        const includeTrialTemplate = {
            model: TrialTemplate,
            as: 'trialTemplate',
            attributes: ['title', 'isProctoring', 'timeLimit']
        };
        const includeTrial = {
            model: Trial,
            as: 'trial',
            attributes: ['started', 'timeLimit', 'questionCount'],
            include: includeTrialTemplate
        };

        const query = _.assign(
            { include: [includeQuestions, includeTrial] },
            options
        );

        return yield TrialToQuestion.findOne(query);
    }

    *answer(answerIds) {
        const attemptQuestion = this.get('attemptQuestion');
        const answered = attemptQuestion.get('answered');

        assert(answered !== ANSWER_CODES.answered, 403, 'Question is already answered', 'QAA');

        if (answerIds) {
            if (!Array.isArray(answerIds)) {
                answerIds = [answerIds];
            }

            const answers = this.get('attemptQuestion.question.answers');
            const correct = AttemptQuestion._isCorrect(answers, answerIds);

            attemptQuestion.set('correct', correct);
        }

        const skippedCode = answered === 0 ? ANSWER_CODES.skipped : answered + 1;

        attemptQuestion.set('answered', answerIds ? ANSWER_CODES.answered : skippedCode);

        yield attemptQuestion.save();
    }

    static _isCorrect(answers, answerIds) {
        for (const answerId of answerIds) {
            assertIsNumber(answerId, 400, 'Answer ID is invalid', 'ANI');
        }

        const correctAnswers = _(answers)
            .filter(answer => answer.correct)
            .map(answer => answer.id)
            .value();

        return _.xor(correctAnswers, answerIds).length ? 0 : 1;
    }

    toJSON() {
        if (!_.get(this._data, 'attemptQuestion')) {
            return {};
        }

        const attemptQuestion = this._data.attemptQuestion.toJSON();
        const result = {
            seq: attemptQuestion.seq,
            questionId: attemptQuestion.questionId,
            answered: _.defaultTo(ANSWER_STATES[attemptQuestion.answered], 'skipped'),
            text: attemptQuestion.question.text,
            type: attemptQuestion.question.type === 0 ? 'one_answer' : 'many_answers'
        };
        const answers = _(attemptQuestion.question.answers)
            .map(answer => _.pick(
                answer,
                ['id', 'text']
            ))
            .shuffle()
            .value();
        const section = _.pick(
            attemptQuestion.question.section,
            ['code', 'title']
        );
        const trial = _.pick(
            attemptQuestion.trial,
            ['started', 'timeLimit', 'questionCount']
        );
        const exam = _.pick(
            _.get(attemptQuestion, 'trial.trialTemplate'),
            ['title', 'isProctoring', 'timeLimit']
        );

        trial.countAnswered = this._data.countAnswered;

        return _.assign(result, { answers, section, trial, exam });
    }
}

module.exports = AttemptQuestion;
