'use strict';

const _ = require('lodash');

const Base = require('models/base');

const {
    Admin,
    Role,
    AdminToRole
} = require('db/postgres');

class Authority extends Base {
    constructor(adminToRoles) {
        super();

        this._adminToRoles = adminToRoles;
    }

    static *find(uid, transaction) {
        const includeAdmin = {
            model: Admin,
            attributes: ['uid'],
            where: { uid }
        };
        const includeRole = {
            model: Role,
            attributes: ['code', 'active'],
            where: { active: 1 }
        };

        const adminToRoles = yield AdminToRole.findAll({
            include: [includeAdmin, includeRole],
            transaction
        });

        return new Authority(adminToRoles);
    }

    getRoles() {
        return this._adminToRoles.map(adminToRole => {
            const role = adminToRole.get('role');

            return role.code;
        });
    }

    _hasRole(roleCode) {
        return _.some(this._adminToRoles, adminToRole => adminToRole.role.code === roleCode);
    }

    get isDeveloper() {
        return this._hasRole('developer');
    }

    get isAnalyst() {
        return this._hasRole('analyst');
    }

    get isAdmin() {
        return this._hasRole('admin');
    }

    get isAssessor() {
        return this._hasRole('assessor');
    }

    get isEditor() {
        return this._hasRole('editor');
    }

    get isSupport() {
        return this._hasRole('support');
    }
}

module.exports = Authority;
